﻿Imports System.Text

Imports TPCANTPHandle = System.UInt16
Imports Peak.Can.IsoTp

''' <summary>
''' Main application that demonstrates ISO-TP API features.
''' </summary>
Public Class FormMain

#Region "Delegates"
    ''' <summary>
    ''' Declare a delegate type to read CANTP messages
    ''' </summary>
    Public Delegate Sub ReadDelegateHandler()
    ''' <summary>
    ''' Declare a delegate type to handle failure while reading CANTP messages
    ''' </summary>
    Public Delegate Sub ReadFailedDelegateHandler(ByVal sts As TPCANTPStatus)
#End Region

#Region "Members"
    ''' <summary>
    ''' Handles of all currently available PCAN-Hardwares.
    ''' </summary>
    Private m_pctpHandles As TPCANTPHandle()
    ''' <summary>
    ''' Connected PCAN ISO-TP Handle (PCANTP_NONEBUS if not connected).
    ''' </summary>
    Private m_pctpHandle As TPCANTPHandle
    ''' <summary>
    ''' DialogBox that handles mapping.
    ''' </summary>
    Private m_dlgMapping As FormMapping
    ''' <summary>
    ''' List of configured PCAN-ISO-TP mappings.
    ''' </summary>
    Private m_mappings As List(Of MappingStatus)
    ''' <summary>
    ''' Collection of MessageStatus used to display each received ISO-TP messages.
    ''' </summary>
    Private m_receiveMsgs As System.Collections.ArrayList
    ''' <summary>
    ''' "Receive CANTP message" Event.
    ''' </summary>
    Private m_receiveEvent As System.Threading.AutoResetEvent
    ''' <summary>
    ''' Read Delegate to read CANTP messages.
    ''' </summary>
    Private m_readDelegate As ReadDelegateHandler
    ''' <summary>
    ''' Read Delegate to read CANTP messages.
    ''' </summary>
    Private m_readFailedDelegate As ReadFailedDelegateHandler
    ''' <summary>
    ''' Thread used to read CANTP messages via events.
    ''' </summary>
    Private m_readThread As System.Threading.Thread
#If _DEBUG_WRITE_CHECK_FILE Then
    ''' <summary>
    ''' Global variable used to know if check file initialization is done
    ''' </summary>
    Private g_FileInitAlreadyDone As Boolean = False
    ''' <summary>
    ''' Global File object
    ''' </summary>
    Private g_CheckFileToWrite As System.IO.StreamWriter
#End If

#End Region

    ''' <summary>
    ''' Default constructor
    ''' </summary>
    Public Sub New()
        InitializeComponent()
        ' Asserts PCAN-ISO-TP library is available.
        Try
            Dim buf As StringBuilder = New StringBuilder(256)
            Dim sts As TPCANTPStatus
            ' Get API version
            sts = CanTpApi.GetValue(CanTpApi.PCANTP_NONEBUS, TPCANTPParameter.PCANTP_PARAM_API_VERSION, buf, CUInt(buf.Capacity))
            Console.WriteLine("PCAN-ISO-TP API version: {0}", buf)
            checkCanTpStatus(CanTpApi.PCANTP_NONEBUS, sts, buf, TPCANTPParameter.PCANTP_PARAM_API_VERSION)
        Catch ex As DllNotFoundException
            MessageBox.Show("Unable to find one of required PCAN libraries: PCANBasic.dll or PCAN-ISO-TP.dll.", "Error!",
                MessageBoxButtons.OK, MessageBoxIcon.Error)
            Environment.Exit(-1)
        End Try
        ' Creates an array with all possible PCAN-ISO-TP Channels.
        m_pctpHandles = New TPCANTPHandle() {
            CanTpApi.PCANTP_ISABUS1,
            CanTpApi.PCANTP_ISABUS2,
            CanTpApi.PCANTP_ISABUS3,
            CanTpApi.PCANTP_ISABUS4,
            CanTpApi.PCANTP_ISABUS5,
            CanTpApi.PCANTP_ISABUS6,
            CanTpApi.PCANTP_ISABUS7,
            CanTpApi.PCANTP_ISABUS8,
            CanTpApi.PCANTP_DNGBUS1,
            CanTpApi.PCANTP_PCIBUS1,
            CanTpApi.PCANTP_PCIBUS2,
            CanTpApi.PCANTP_PCIBUS3,
            CanTpApi.PCANTP_PCIBUS4,
            CanTpApi.PCANTP_PCIBUS5,
            CanTpApi.PCANTP_PCIBUS6,
            CanTpApi.PCANTP_PCIBUS7,
            CanTpApi.PCANTP_PCIBUS8,
            CanTpApi.PCANTP_PCIBUS9,
            CanTpApi.PCANTP_PCIBUS10,
            CanTpApi.PCANTP_PCIBUS11,
            CanTpApi.PCANTP_PCIBUS12,
            CanTpApi.PCANTP_PCIBUS13,
            CanTpApi.PCANTP_PCIBUS14,
            CanTpApi.PCANTP_PCIBUS15,
            CanTpApi.PCANTP_PCIBUS16,
            CanTpApi.PCANTP_USBBUS1,
            CanTpApi.PCANTP_USBBUS2,
            CanTpApi.PCANTP_USBBUS3,
            CanTpApi.PCANTP_USBBUS4,
            CanTpApi.PCANTP_USBBUS5,
            CanTpApi.PCANTP_USBBUS6,
            CanTpApi.PCANTP_USBBUS7,
            CanTpApi.PCANTP_USBBUS8,
            CanTpApi.PCANTP_USBBUS9,
            CanTpApi.PCANTP_USBBUS10,
            CanTpApi.PCANTP_USBBUS11,
            CanTpApi.PCANTP_USBBUS12,
            CanTpApi.PCANTP_USBBUS13,
            CanTpApi.PCANTP_USBBUS14,
            CanTpApi.PCANTP_USBBUS15,
            CanTpApi.PCANTP_USBBUS16,
            CanTpApi.PCANTP_PCCBUS1,
            CanTpApi.PCANTP_PCCBUS2,
            CanTpApi.PCANTP_LANBUS1,
            CanTpApi.PCANTP_LANBUS2,
            CanTpApi.PCANTP_LANBUS3,
            CanTpApi.PCANTP_LANBUS4,
            CanTpApi.PCANTP_LANBUS5,
            CanTpApi.PCANTP_LANBUS6,
            CanTpApi.PCANTP_LANBUS7,
            CanTpApi.PCANTP_LANBUS8,
            CanTpApi.PCANTP_LANBUS9,
            CanTpApi.PCANTP_LANBUS10,
            CanTpApi.PCANTP_LANBUS11,
            CanTpApi.PCANTP_LANBUS12,
            CanTpApi.PCANTP_LANBUS13,
            CanTpApi.PCANTP_LANBUS14,
            CanTpApi.PCANTP_LANBUS15,
            CanTpApi.PCANTP_LANBUS16
        }
        ' Initializes array of received ISO-TP messages
        m_receiveMsgs = New System.Collections.ArrayList()
        ' Initializes list of configured mappings
        m_mappings = New List(Of MappingStatus)()
        ' Creates the delegate used to message reading
        m_readDelegate = AddressOf readMessages
        ' Creates the delegate used when reading-thread fails
        m_readFailedDelegate = AddressOf readThreadFailed
        ' Creates the event used to signalize incomming messages 
        m_receiveEvent = New System.Threading.AutoResetEvent(False)
        ' Initializes UI components
        initializeUiConnection()
        initializeUiTabMessages()
        initializeUiTabParameters()
        ' Initializes the "mapping" dialog once and for all
        m_dlgMapping = New FormMapping()

        textBoxCanFdBitrate.Text = "f_clock_mhz=20, nom_brp=5, nom_tseg1=2, nom_tseg2=1, nom_sjw=1, data_brp=2, data_tseg1=3, data_tseg2=1, data_sjw=1"
        checkBoxCanFd_CheckedChanged(Me, EventArgs.Empty) ' initialization Of UI components For FD messages
    End Sub

#Region "Main Form event handlers"
    ''' <summary>
    ''' Event handler called when main window is first shown.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub FormMain_Shown(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles MyBase.Shown
        ' UI Components are all initialized, refresh mappings' tab.
        mappingsChanged()
        ' Set connection status to false to update all UI's components
        setConnectionStatus(False)
    End Sub

    ''' <summary>
    ''' Event handler called when main window is closed.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub FormMain_FormClosed(ByVal sender As System.Object, ByVal e As System.Windows.Forms.FormClosedEventArgs) Handles MyBase.FormClosed
        Dim sts As TPCANTPStatus
        ' Disconnect all PCAN-ISO-TP connections.
        sts = CanTpApi.Uninitialize(CanTpApi.PCANTP_NONEBUS)
        checkCanTpStatus(m_pctpHandle, sts)
    End Sub
#End Region

#Region "Connection > Event Handlers"
    ''' <summary>
    ''' Event handler called when button "Hardware Refresh" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonHwRefresh_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonHwRefresh.Click
        Dim sts As TPCANTPStatus
        Dim cbItem As ComboBoxItem                 ' an item of the "channel" comboBox
        Dim pctpHandle As TPCANTPHandle            ' temporary PCAN ISO-TP handle
        Dim pctpHandleSelected As TPCANTPHandle    ' store the currently selected channel in the "channel" comboBox
        Dim iBuffer As UInt32                      ' temporary buffer to retrieve the condition of a channel

        'Get selected item first (to set it back later).
        If comboBoxChannel.SelectedItem IsNot Nothing AndAlso TypeOf comboBoxChannel.SelectedItem Is ComboBoxItem Then
            pctpHandleSelected = DirectCast(DirectCast(comboBoxChannel.SelectedItem, ComboBoxItem).Data, TPCANTPHandle)
        Else
            pctpHandleSelected = CanTpApi.PCANTP_NONEBUS
        End If
        ' Clears the "Channel" comboBox and fill it again with 
        '  all the PCAN-ISO-TP handles (non-plug'n'play hardware and
        '  the detected Plug'n'Play hardware).
        comboBoxChannel.Items.Clear()
        ' Loop through all known PCAN-ISO-TP handles.
        For i As Integer = 0 To m_pctpHandles.Length - 1
            pctpHandle = m_pctpHandles(i)
            ' Initializes an item for the comboBox.
            cbItem = New ComboBoxItem(CanTpUtils.GetChannelName(pctpHandle), pctpHandle)
            ' Automatically add non-plug'n'play handles.
            If pctpHandle <= CanTpApi.PCANTP_DNGBUS1 Then
                comboBoxChannel.Items.Add(cbItem)
            Else
                ' Check the availalility of plug'n'play handles.
                sts = CanTpApi.GetValue(pctpHandle, TPCANTPParameter.PCANTP_PARAM_CHANNEL_CONDITION, iBuffer, System.Runtime.InteropServices.Marshal.SizeOf(iBuffer))
                If sts = TPCANTPStatus.PCANTP_ERROR_OK AndAlso
                        ((iBuffer And CanTpApi.PCANTP_CHANNEL_AVAILABLE) = CanTpApi.PCANTP_CHANNEL_AVAILABLE) Then
                    ' Channel is available, add it to the comboBox.
                    comboBoxChannel.Items.Add(cbItem)
                End If
            End If
            ' Select the handle if it was the previously selected one.
            If pctpHandle = pctpHandleSelected Then
                comboBoxChannel.SelectedIndex = comboBoxChannel.Items.Count - 1
            End If
        Next
    End Sub
    ''' <summary>
    ''' Event handler called when button "Initialize" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonInit_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonInit.Click
        Dim pctpHandle As TPCANTPHandle
        Dim baudrate As TPCANTPBaudrate
        Dim hwType As TPCANTPHWType
        Dim ioPort As UInteger
        Dim interrupt As Byte

        If comboBoxChannel.SelectedItem Is Nothing Then
            Return
        End If
        ' Gets the selected ISO-TP channel.
        pctpHandle = DirectCast(DirectCast(comboBoxChannel.SelectedItem, ComboBoxItem).Data, TPCANTPHandle)
        ' Gets the selected baudrate.
        baudrate = DirectCast(DirectCast(comboBoxBaudrate.SelectedItem, ComboBoxItem).Data, TPCANTPBaudrate)
        ' Read the HarwareType, IO port and Interrupt only 
        ' if the selected device is not plug'n'play.
        hwType = 0
        ioPort = 0
        interrupt = 0
        If pctpHandle <= CanTpApi.PCANTP_DNGBUS1 Then
            hwType = DirectCast(DirectCast(comboBoxHwType.SelectedItem, ComboBoxItem).Data, TPCANTPHWType)
            ioPort = CUInt(DirectCast(comboBoxIoPort.SelectedItem, ComboBoxItem).Data)
            interrupt = CByte(DirectCast(comboBoxInterrupt.SelectedItem, ComboBoxItem).Data)
        End If
        ' Connects to the selected PCAN-ISO-TP channel.
        connect(pctpHandle, baudrate, hwType, ioPort, interrupt)
    End Sub
    ''' <summary>
    ''' Event handler called when button "Release" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonRelease_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonRelease.Click
        'Disconnects the connected PCAN-ISO-TP channel.
        disconnect()
    End Sub
    ''' <summary>
    ''' Event handler called when selection of comboBox "channel handles" is changed.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub comboBoxChannel_SelectedIndexChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles comboBoxChannel.SelectedIndexChanged
        Dim pctpHandle As TPCANTPHandle
        Dim isNotPnp As Boolean

        ' Enables "Initialize" button if a channel is selected.
        buttonInit.Enabled = (comboBoxChannel.SelectedItem IsNot Nothing)
        ' Aborts if no channel is selected.
        If comboBoxChannel.SelectedItem Is Nothing Then
            Return
        End If
        ' Gets the handle from the selected item.
        pctpHandle = DirectCast(DirectCast(comboBoxChannel.SelectedItem, ComboBoxItem).Data, TPCANTPHandle)
        ' Determines if the handle belong to a non-plug'n'play hardware.
        isNotPnp = (pctpHandle <= CanTpApi.PCANTP_DNGBUS1)
        ' Activates/deactivates configuration controls according to the kind of hardware.
        comboBoxHwType.Enabled = isNotPnp
        comboBoxIoPort.Enabled = isNotPnp
        comboBoxInterrupt.Enabled = isNotPnp
    End Sub
#End Region

#Region "Parameters > Event Handlers"
    ''' <summary>
    ''' Event handler called when button "Parameters > Get" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonParamGet_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonParamGet.Click
        Dim param As TPCANTPParameter
        Dim sts As TPCANTPStatus
        Dim iBuf As UInteger
        Dim strBuf As StringBuilder
        Dim canHandle As TPCANTPHandle

        ' Checks selection is valid.
        If comboBoxParameter.SelectedItem Is Nothing OrElse Not (TypeOf comboBoxParameter.SelectedItem Is ComboBoxItem) Then
            Return
        End If
        ' Gets selected parameter
        param = DirectCast(DirectCast(comboBoxParameter.SelectedItem, ComboBoxItem).Data, TPCANTPParameter)
        iBuf = 0
        Select Case param
            ' Retrieves the API version.
            Case TPCANTPParameter.PCANTP_PARAM_API_VERSION
                strBuf = New StringBuilder(256)
                sts = CanTpApi.GetValue(m_pctpHandle, param, strBuf, CUInt(strBuf.Capacity))
                checkCanTpStatus(m_pctpHandle, sts, strBuf, param)
                If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
                    includeTextMessage("ISO-TP API version:" + strBuf.ToString)
                End If
                Exit Select
                ' Reads the ISO-TP BlockSize (BS) parameter (number of Consecutive Frame between Flow Control frames).
            Case TPCANTPParameter.PCANTP_PARAM_BLOCK_SIZE
                sts = CanTpApi.GetValue(m_pctpHandle, param, iBuf, System.Runtime.InteropServices.Marshal.SizeOf(iBuf))
                checkCanTpStatus(m_pctpHandle, sts, iBuf, param)
                If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
                    includeTextMessage(String.Format("ISO-TP Block Size parameter is {0} ({0:X2}h).", iBuf))
                End If
                Exit Select
                ' Reads if CAN DATA Padding is enabled (if enabled, all segmented CAN frames have a DLC of 8).
            Case TPCANTPParameter.PCANTP_PARAM_CAN_DATA_PADDING
                sts = CanTpApi.GetValue(m_pctpHandle, param, iBuf, System.Runtime.InteropServices.Marshal.SizeOf(iBuf))
                checkCanTpStatus(m_pctpHandle, sts, iBuf, param)
                If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
                    includeTextMessage(String.Format("CAN Data Padding is {0} ({1}).",
                        If((iBuf = CanTpApi.PCANTP_CAN_DATA_PADDING_NONE), "disabled", "enabled"), iBuf))
                End If
                Exit Select
                ' Reads the channel's state.
            Case TPCANTPParameter.PCANTP_PARAM_CHANNEL_CONDITION
                If comboBoxChannel.SelectedItem IsNot Nothing AndAlso TypeOf comboBoxChannel.SelectedItem Is ComboBoxItem Then
                    canHandle = DirectCast(DirectCast(comboBoxChannel.SelectedItem, ComboBoxItem).Data, TPCANTPHandle)
                Else
                    canHandle = m_pctpHandle
                End If
                sts = CanTpApi.GetValue(canHandle, param, iBuf, System.Runtime.InteropServices.Marshal.SizeOf(iBuf))
                checkCanTpStatus(m_pctpHandle, sts, iBuf, param)
                If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
                    Select Case iBuf
                        Case CanTpApi.PCANTP_CHANNEL_AVAILABLE
                            includeTextMessage(String.Format("Channel {0:X2}h is available ({1}).", canHandle, iBuf))
                            Exit Select
                        Case CanTpApi.PCANTP_CHANNEL_UNAVAILABLE
                            includeTextMessage(String.Format("Channel {0:X2}h is unavailable ({1}).", canHandle, iBuf))
                            Exit Select
                        Case CanTpApi.PCANTP_CHANNEL_OCCUPIED
                            includeTextMessage(String.Format("Channel {0:X2}h is occupied ({1}).", canHandle, iBuf))
                            Exit Select
                        Case Else
                            includeTextMessage(String.Format("Channel {0:X2}h is in an unknown state ({1}).", canHandle, iBuf))
                            Exit Select
                    End Select
                End If
                Exit Select
                ' Reads if Debug mode is enabled (if enabled, CAN Tx/Rx frames are echoed on console output).
            Case TPCANTPParameter.PCANTP_PARAM_DEBUG
                sts = CanTpApi.GetValue(m_pctpHandle, param, iBuf, System.Runtime.InteropServices.Marshal.SizeOf(iBuf))
                checkCanTpStatus(m_pctpHandle, sts, iBuf, param)
                If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
                    includeTextMessage(String.Format("Debug mode 'CAN frames on console output' is {0} ({1}).",
                        If((iBuf = CanTpApi.PCANTP_DEBUG_NONE), "disabled", "enabled"), iBuf))
                End If
                Exit Select
                ' Reads if Message Pending Mode is enabled 
                '  (if enabled, CAN ISO-TP messages with type PCANTP_MESSAGE_INDICATION 
                ' can be read when the first frame of a segmented message is received).
            Case TPCANTPParameter.PCANTP_PARAM_MSG_PENDING
                sts = CanTpApi.GetValue(m_pctpHandle, param, iBuf, System.Runtime.InteropServices.Marshal.SizeOf(iBuf))
                checkCanTpStatus(m_pctpHandle, sts, iBuf, param)
                If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
                    includeTextMessage(String.Format("Notifications of incoming CAN-ISO-TP messages are {0} ({1}).",
                        If((iBuf = CanTpApi.PCANTP_MSG_PENDING_HIDE), "disabled", "enabled"), iBuf))
                End If
                Exit Select
                ' Reads the padding value used when "CAN DATA Padding" is enabled.
            Case TPCANTPParameter.PCANTP_PARAM_PADDING_VALUE
                sts = CanTpApi.GetValue(m_pctpHandle, param, iBuf, System.Runtime.InteropServices.Marshal.SizeOf(iBuf))
                checkCanTpStatus(m_pctpHandle, sts, iBuf, param)
                If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
                    includeTextMessage(String.Format("Value for 'CAN Data Padding' is {0:X2}h ({0}).", iBuf))
                End If
                Exit Select
                ' Reads the ISO-TP default priority value for normal fixed, mixed And enhanced addressing
            Case TPCANTPParameter.PCANTP_PARAM_J1939_PRIORITY
                sts = CanTpApi.GetValue(m_pctpHandle, param, iBuf, System.Runtime.InteropServices.Marshal.SizeOf(iBuf))
                checkCanTpStatus(m_pctpHandle, sts, iBuf, param)
                If (sts = TPCANTPStatus.PCANTP_ERROR_OK) Then
                    includeTextMessage(String.Format("The default priority value for normal fixed, mixed and enhanced addressing is {0:X2}h ({0}).", iBuf))
                End If
                Exit Select
                ' Reads the ISO-TP default DLC to use when transmitting messages with CAN FD
                    Case TPCANTPParameter.PCANTP_PARAM_CAN_TX_DL
                sts = CanTpApi.GetValue(m_pctpHandle, param, iBuf, System.Runtime.InteropServices.Marshal.SizeOf(iBuf))
                checkCanTpStatus(m_pctpHandle, sts, iBuf, param)
                If (sts = TPCANTPStatus.PCANTP_ERROR_OK) Then
                    includeTextMessage(String.Format("The default DLC to use when transmitting messages with CAN FD is {0:X2}h ({0}).", iBuf))
                End If
                Exit Select
                ' Reads the receive-event handle (if receive-event is disabled, 0 is returned).
                        Case TPCANTPParameter.PCANTP_PARAM_RECEIVE_EVENT
                sts = CanTpApi.GetValue(m_pctpHandle, param, iBuf, System.Runtime.InteropServices.Marshal.SizeOf(iBuf))
                checkCanTpStatus(m_pctpHandle, sts, iBuf, param)
                If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
                    includeTextMessage(String.Format("Receive event is {0} (event handle is {1:X}h).",
                        If((iBuf = 0), "disabled", "enabled"), iBuf))
                End If
                Exit Select
                ' Reads the ISO-TP SeparationTime (STmin) parameter 
                '  (minimum time to wait between Consecutive Frames transmission).
            Case TPCANTPParameter.PCANTP_PARAM_SEPARATION_TIME
                sts = CanTpApi.GetValue(m_pctpHandle, param, iBuf, System.Runtime.InteropServices.Marshal.SizeOf(iBuf))
                checkCanTpStatus(m_pctpHandle, sts, iBuf, param)
                If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
                    If iBuf < &H80 Then
                        includeTextMessage(String.Format("ISO-TP Separation Time (STmin) is {0}ms ({0:X2}h).", iBuf))
                    ElseIf iBuf > &HF0 And iBuf < &HFA Then
                        includeTextMessage(String.Format("ISO-TP Separation Time (STmin) is {0}µs ({0:X2}h).", (iBuf - &HF0) * 100))
                    Else
                        includeTextMessage(String.Format("ISO-TP Separation Time (STmin) uses a reserved value of ISO 15765 ({0:X2}h).", iBuf))
                    End If
                End If
                Exit Select
                ' Reads the ISO-TP FC.Wait frame transmissions (N_WFTMax) parameter 
                '  (maximum number of consecutive Flow Control frames with the Wait status after which transmission is aborted).
            Case TPCANTPParameter.PCANTP_PARAM_WFT_MAX
                sts = CanTpApi.GetValue(m_pctpHandle, param, iBuf, System.Runtime.InteropServices.Marshal.SizeOf(iBuf))
                checkCanTpStatus(m_pctpHandle, sts, iBuf, param)
                If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
                    includeTextMessage(String.Format("Maximum number of FC.Wait frame transmissions (N_WFTMax) is {0}.", iBuf))
                End If
                Exit Select
                ' Generic reading request of a parameter (should not occur).
            Case Else
                sts = CanTpApi.GetValue(m_pctpHandle, param, iBuf, System.Runtime.InteropServices.Marshal.SizeOf(iBuf))
                checkCanTpStatus(m_pctpHandle, sts, iBuf, param)
                If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
                    includeTextMessage(String.Format("Unknown parameter '{0}' is {1:X2}h", param, iBuf))
                End If
                Exit Select
        End Select
    End Sub
    ''' <summary>
    ''' Event handler called when button "Parameters > Set" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonParamSet_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonParamSet.Click
        Dim param As TPCANTPParameter
        Dim sts As TPCANTPStatus
        Dim iBuf As UInteger
        ' Checks selection is valid.
        If comboBoxParameter.SelectedItem Is Nothing OrElse Not (TypeOf comboBoxParameter.SelectedItem Is ComboBoxItem) Then
            Return
        End If
        ' Sets selected parameter
        param = DirectCast(DirectCast(comboBoxParameter.SelectedItem, ComboBoxItem).Data, TPCANTPParameter)
        Select Case param
            ' Sets the ISO-TP BlockSize (BS) parameter (number of Consecutive Frame between Flow Control frames).
            Case TPCANTPParameter.PCANTP_PARAM_BLOCK_SIZE
                iBuf = Convert.ToUInt32(numericUpDownParamValue.Value)
                sts = CanTpApi.SetValue(m_pctpHandle, param, iBuf, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuf), UInteger))
                Exit Select
                ' Sets if CAN DATA Padding is enabled (if enabled, all segmented CAN frames have a DLC of 8).
            Case TPCANTPParameter.PCANTP_PARAM_CAN_DATA_PADDING
                iBuf = If(radioButtonParamActive.Checked, CanTpApi.PCANTP_CAN_DATA_PADDING_ON, CanTpApi.PCANTP_CAN_DATA_PADDING_NONE)
                sts = CanTpApi.SetValue(m_pctpHandle, param, iBuf, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuf), UInteger))
                Exit Select
                ' Sets if Debug mode is enabled (if enabled, CAN Tx/Rx frames are echoed on console output).
            Case TPCANTPParameter.PCANTP_PARAM_DEBUG
                iBuf = If(radioButtonParamActive.Checked, CanTpApi.PCANTP_DEBUG_CAN, CanTpApi.PCANTP_DEBUG_NONE)
                sts = CanTpApi.SetValue(m_pctpHandle, param, iBuf, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuf), UInteger))
                Exit Select
                ' Sets if Message Pending Mode is enabled 
                '  (if enabled, CAN ISO-TP messages with type PCANTP_MESSAGE_INDICATION 
                '  can be read when the first frame of a segmented message is received).
            Case TPCANTPParameter.PCANTP_PARAM_MSG_PENDING
                iBuf = If(radioButtonParamActive.Checked, CanTpApi.PCANTP_MSG_PENDING_SHOW, CanTpApi.PCANTP_MSG_PENDING_HIDE)
                sts = CanTpApi.SetValue(m_pctpHandle, param, iBuf, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuf), UInteger))
                Exit Select
                ' Sets the padding value used when "CAN DATA Padding" is enabled.
            Case TPCANTPParameter.PCANTP_PARAM_PADDING_VALUE
                iBuf = Convert.ToUInt32(numericUpDownParamValue.Value)
                sts = CanTpApi.SetValue(m_pctpHandle, param, iBuf, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuf), UInteger))
                Exit Select
                 ' Sets the priority value used when "J1939 Priority" Is enabled.
            Case TPCANTPParameter.PCANTP_PARAM_J1939_PRIORITY
                iBuf = Convert.ToUInt32(numericUpDownParamValue.Value)
                sts = CanTpApi.SetValue(m_pctpHandle, param, iBuf, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuf), UInteger))
                Exit Select
                ' Sets the DLC value used when "Data Length Code (DLC)" Is enabled.
            Case TPCANTPParameter.PCANTP_PARAM_CAN_TX_DL
                iBuf = Convert.ToUInt32(numericUpDownParamValue.Value)
                sts = CanTpApi.SetValue(m_pctpHandle, param, iBuf, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuf), UInteger))
                Exit Select
               ' Sets the ISO-TP SeparationTime (STmin) parameter 
                '  (minimum time to wait between Consecutive Frames transmission).
            Case TPCANTPParameter.PCANTP_PARAM_SEPARATION_TIME
                iBuf = Convert.ToUInt32(numericUpDownParamValue.Value)
                sts = CanTpApi.SetValue(m_pctpHandle, param, iBuf, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuf), UInteger))
                Exit Select
                ' Sets the ISO-TP FC.Wait frame transmissions (N_WFTMax) parameter 
                '  (maximum number of consecutive Flow Control frames with the Wait status after which transmission is aborted).
            Case TPCANTPParameter.PCANTP_PARAM_WFT_MAX
                iBuf = Convert.ToUInt32(numericUpDownParamValue.Value)
                sts = CanTpApi.SetValue(m_pctpHandle, param, iBuf, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuf), UInteger))
                Exit Select
                ' Unhandled parameter (should not occur).
            Case Else
                includeTextMessage(String.Format("Unknown parameter '{0}', setting aborted.", param))
                sts = TPCANTPStatus.PCANTP_ERROR_WRONG_PARAM
                Exit Select
        End Select
        checkCanTpStatus(m_pctpHandle, sts, param, iBuf)
        ' Outputs the new parameter value.
        buttonParamGet_Click(Me, EventArgs.Empty)
    End Sub
    ''' <summary>
    ''' Event handler called when button "Parameters > Clear Information" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonParamInfoClear_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonParamInfoClear.Click
        ' Clears the information of the Information List-Box 
        listBoxParamInfo.Items.Clear()
    End Sub
    ''' <summary>
    ''' Event handler called when button "Parameters > Reset" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonParamReset_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonParamReset.Click
        Dim sts As TPCANTPStatus

        ' Resets the receive and transmit queues of a PCAN Channel.
        sts = CanTpApi.Reset(m_pctpHandle)
        checkCanTpStatus(m_pctpHandle, sts)
        If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
            includeTextMessage("Receive and transmit queues successfully reset.")
        End If
    End Sub
    ''' <summary>
    ''' Event handler called when button "Parameters > Status" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonParamStatus_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonParamStatus.Click
        Dim stsResult As TPCANTPStatus
        Dim errorName As String

        ' Gets the current BUS status of a PCAN Channel.
        stsResult = CanTpApi.GetStatus(m_pctpHandle)
        ' Formats on status
        Select Case stsResult
            Case TPCANTPStatus.PCANTP_ERROR_NOT_INITIALIZED
                errorName = "PCANTP_ERROR_NOT_INITIALIZED"
                Exit Select
            Case TPCANTPStatus.PCANTP_ERROR_BUSLIGHT
                errorName = "PCANTP_ERROR_BUSLIGHT"
                Exit Select
            Case TPCANTPStatus.PCANTP_ERROR_BUSHEAVY ' TPCANTPStatus.PCANTP_ERROR_BUSWARNING
                errorName = "PCANTP_ERROR_BUSHEAVY"
                Exit Select
            Case TPCANTPStatus.PCANTP_ERROR_BUSOFF
                errorName = "PCANTP_ERROR_BUSOFF"
                Exit Select
            Case TPCANTPStatus.PCANTP_ERROR_OK
                errorName = "PCANTP_ERROR_OK"
                Exit Select
            Case Else
                errorName = "See Documentation"
                Exit Select
        End Select
        ' Displays Message
        includeTextMessage(String.Format("Channel status: {0} ({1:X}h)", errorName, stsResult))
    End Sub
    ''' <summary>
    ''' Event handler called when button "Parameters > Version" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonParamVersion_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonParamVersion.Click
        Dim sts As TPCANTPStatus
        Dim strTemp As StringBuilder

        ' Gets the vesion of the ISO-TP API
        strTemp = New StringBuilder(256)
        sts = CanTpApi.GetValue(CanTpApi.PCANTP_NONEBUS,
            TPCANTPParameter.PCANTP_PARAM_API_VERSION, strTemp, CUInt(strTemp.Capacity))
        checkCanTpStatus(CanTpApi.PCANTP_NONEBUS, sts, strTemp, TPCANTPParameter.PCANTP_PARAM_API_VERSION)
        If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
            includeTextMessage("API Version: " + strTemp.ToString())
        End If
    End Sub
    ''' <summary>
    ''' Event handler called when the selection of the comboBox "Parameters" is changed.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub comboBoxParameter_SelectedIndexChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles comboBoxParameter.SelectedIndexChanged
        Dim param As TPCANTPParameter
        Dim isRadioEnabled, isNudEnabled, isSetEnabled, isGetEnabled As Boolean
        Dim nudMaxValue, nudMinValue, nudDefaultValue As UInteger

        nudMinValue = &H0
        nudMaxValue = &HFF
        nudDefaultValue = &H0
        isRadioEnabled = False
        isNudEnabled = False
        isSetEnabled = False
        isGetEnabled = False
        If comboBoxParameter.SelectedItem IsNot Nothing AndAlso
            (TypeOf comboBoxParameter.SelectedItem Is ComboBoxItem) Then
            isGetEnabled = isConnected()
            ' Activates/deactivates controls according to the selected parameter 
            param = DirectCast(DirectCast(comboBoxParameter.SelectedItem, ComboBoxItem).Data, TPCANTPParameter)
            Select Case param
                Case TPCANTPParameter.PCANTP_PARAM_API_VERSION
                    isGetEnabled = True
                    Exit Select
                Case TPCANTPParameter.PCANTP_PARAM_BLOCK_SIZE
                    isNudEnabled = True
                    isSetEnabled = isConnected()
                    Exit Select
                Case TPCANTPParameter.PCANTP_PARAM_CAN_DATA_PADDING
                    isRadioEnabled = True
                    isSetEnabled = isConnected()
                    Exit Select
                Case TPCANTPParameter.PCANTP_PARAM_CHANNEL_CONDITION
                    isGetEnabled = True
                    Exit Select
                Case TPCANTPParameter.PCANTP_PARAM_DEBUG
                    isRadioEnabled = True
                    isSetEnabled = isConnected()
                    Exit Select
                Case TPCANTPParameter.PCANTP_PARAM_MSG_PENDING
                    isRadioEnabled = True
                    isSetEnabled = isConnected()
                    Exit Select
                Case TPCANTPParameter.PCANTP_PARAM_PADDING_VALUE
                    isNudEnabled = True
                    isSetEnabled = isConnected()
                    Exit Select
                Case TPCANTPParameter.PCANTP_PARAM_J1939_PRIORITY
                    isNudEnabled = True
                    nudMaxValue = &H7
                    nudDefaultValue = &H6
                    isSetEnabled = isConnected()
                    Exit Select
                Case TPCANTPParameter.PCANTP_PARAM_CAN_TX_DL
                    isNudEnabled = True
                    nudMaxValue = &HF
                    nudMinValue = &H8
                    nudDefaultValue = &H8
                    isSetEnabled = isConnected()
                    Exit Select
                Case TPCANTPParameter.PCANTP_PARAM_RECEIVE_EVENT
                    Exit Select
                Case TPCANTPParameter.PCANTP_PARAM_SEPARATION_TIME
                    isNudEnabled = True
                    isSetEnabled = isConnected()
                    Exit Select
                Case TPCANTPParameter.PCANTP_PARAM_WFT_MAX
                    isGetEnabled = True
                    isNudEnabled = True
                    isSetEnabled = True
                    nudMaxValue = &HFFFF
                    Exit Select
            End Select
        End If
        ' Activates/deactivates controls
        buttonParamGet.Enabled = isGetEnabled
        buttonParamSet.Enabled = isSetEnabled
        radioButtonParamActive.Enabled = isRadioEnabled
        radioButtonParamInactive.Enabled = isRadioEnabled
        numericUpDownParamValue.Enabled = isNudEnabled
        numericUpDownParamValue.Maximum = nudMaxValue
        numericUpDownParamValue.Minimum = nudMinValue
        numericUpDownParamValue.Value = nudDefaultValue
    End Sub
#End Region

#Region "Mapping > Event Handlers"
    ''' <summary>
    ''' Event handler called when button "Add Mapping" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonMappingAdd_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonMappingAdd.Click
        ' Prepares dialogBox's UI components with a default mapping.
        m_dlgMapping.initializeMapping(Nothing)
        ' Shows dialog
        If m_dlgMapping.ShowDialog() = DialogResult.OK Then
            ' Clones the mapping from the dialogBox and adds it to the API.
            mappingsAdd(m_dlgMapping.m_mapping.Clone())
            ' Refreshes UI dealing with mappings.
            mappingsChanged()
        End If
    End Sub
    ''' <summary>
    ''' Event handler called when button "Delete Mapping" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonMappingDel_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonMappingDel.Click
        ' Asserts the selection is valid
        If listViewMappings.SelectedItems.Count > 0 AndAlso
            (TypeOf listViewMappings.SelectedItems(0).Tag Is MappingStatus) Then
            Dim mapping As MappingStatus
            Dim sts As TPCANTPStatus

            ' Gets the selected mapping.
            mapping = DirectCast(listViewMappings.SelectedItems(0).Tag, MappingStatus)
            ' Prevent deletion of "automatic mapping" 
            '  (those mappings do not exist in the ISO-TP API, they are displayed
            '  as a tutorial purpose)
            If mapping.IsAutomatic Then
                Return
            End If
            ' Removes the mapping from the API (via the CAN ID that defines it uniquely)
            sts = CanTpApi.RemoveMapping(m_pctpHandle, mapping.m_canId)
            checkCanTpStatus(m_pctpHandle, sts, mapping.m_canId)
            If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
                ' Removes the mapping from the internal list of mappings
                m_mappings.Remove(mapping)
                ' Removes the corresponding listViewItem 
                listViewMappings.Items.Remove(listViewMappings.SelectedItems(0))
            End If
        End If
    End Sub
    ''' <summary>
    ''' Event handler called when button "Mapping example" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonMappingSample_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonMappingSample.Click
        Dim mapping As MappingStatus
        Dim clientAddr As Byte            ' Tester Client address
        Dim ecuAddr As Byte               ' ECU address
        Dim funcAddr As Byte              ' Functional address
        Dim canIdCli2Ecu As UInteger      ' CAN ID used to communicate from Client to ECU
        Dim canIdEcu2Cli As UInteger      ' CAN ID used to communicate from ECU to Client
        Dim canIdCliFunc As UInteger      ' CAN ID used to communicate from Client to any ECUs

        ' The sample defines 11 bits CAN ID, normal format addressing mappings 
        '  (diagnostic message will be mandatory since normal format addressing is used)
        clientAddr = &HF1
        ecuAddr = &H13
        funcAddr = &H20
        canIdCli2Ecu = &HA1
        canIdEcu2Cli = &HA2
        canIdCliFunc = &HA3
        ' Defines a mapping to allow communication from client to ECU (a.k.a request).
        mapping = New MappingStatus(canIdCli2Ecu, canIdEcu2Cli,
            TPCANTPIdType.PCANTP_ID_CAN_11BIT,
            TPCANTPFormatType.PCANTP_FORMAT_NORMAL,
            TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC,
            TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL,
            clientAddr, ecuAddr, &H0)
        mappingsAdd(mapping)
        ' Defines a mapping to allow communication from ECU to client (a.k.a response).
        mapping = New MappingStatus(canIdEcu2Cli, canIdCli2Ecu,
            TPCANTPIdType.PCANTP_ID_CAN_11BIT,
            TPCANTPFormatType.PCANTP_FORMAT_NORMAL,
            TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC,
            TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL,
            ecuAddr, clientAddr, &H0)
        mappingsAdd(mapping)
        ' define mapping to allow communication from client to any ECUs (a.k.a functional request).
        mapping = New MappingStatus(canIdCliFunc, CanTpApi.CAN_ID_NO_MAPPING,
            TPCANTPIdType.PCANTP_ID_CAN_11BIT,
            TPCANTPFormatType.PCANTP_FORMAT_NORMAL,
            TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC,
            TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL,
            clientAddr, funcAddr, &H0)
        mappingsAdd(mapping)
        ' Refreshes all UIs dealing with mappings.
        mappingsChanged()
    End Sub

    ''' <summary>
    ''' Event handler called when selection of listview "mapping" is changed.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub listViewMappings_SelectedIndexChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles listViewMappings.SelectedIndexChanged
        Dim enabled As Boolean

        ' Enable/disable "remove mapping" button based on listView's selection 
        enabled = (listViewMappings.SelectedItems.Count > 0)
        If enabled Then
            ' Also do not allow removal of "automatic mapping" (since its a cosmetic display).
            Dim mapping As MappingStatus = DirectCast(listViewMappings.SelectedItems(0).Tag, MappingStatus)
            If (mapping IsNot Nothing AndAlso mapping.IsAutomatic) Then
                enabled = False
            End If
        End If
        ' Updates button status.
        buttonMappingDel.Enabled = enabled
    End Sub
#End Region

#Region "Messages > Read Event Handlers"
    ''' <summary>
    ''' Event handler called when button "Clear message" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonMsgClear_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonMsgClear.Click
        Dim sts As TPCANTPStatus

        ' Clears ISO-TP Rx/Tx queues only if channel is connected.
        If isConnected() Then
            sts = CanTpApi.Reset(m_pctpHandle)
            checkCanTpStatus(m_pctpHandle, sts)
        End If
        ' Clears "received messages" listview and internal list of received messages.
        SyncLock m_receiveMsgs.SyncRoot
            m_receiveMsgs.Clear()
            listViewMsgs.Items.Clear()
        End SyncLock
    End Sub
    ''' <summary>
    ''' Event handler called when button "Read message" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonMsgRead_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonMsgRead.Click
        ' Reads ISO-TP messages.
        readMessages()
    End Sub
    ''' <summary>
    ''' Event handler called when the value of the checkBox "Show Timestamp as period" is changed.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub checkBoxMsgShowPeriod_CheckedChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles checkBoxMsgShowPeriod.CheckedChanged
        If m_receiveMsgs Is Nothing Then
            Return
        End If
        ' According to the check-value of this checkbox,
        '  the received time of a messages will be interpreted as a
        '  period (time between the two last messages) or as time-stamp
        '  (the elapsed time since windows was started).
        SyncLock m_receiveMsgs.SyncRoot
            For Each msg As MessageStatus In m_receiveMsgs
                msg.ShowingPeriod = checkBoxMsgShowPeriod.Checked
            Next
        End SyncLock
    End Sub

    ''' <summary>
    ''' Event handler called when radioButton "Reading Type" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub radioButtonMsgRead_CheckedChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles radioButtonMsgTimer.CheckedChanged,
        radioButtonMsgEvent.CheckedChanged, radioButtonMsgManual.CheckedChanged
        ' Terminates "read" thread if it exists.
        If m_readThread IsNot Nothing Then
            m_readThread.Abort()
            m_readThread.Join()
            m_readThread = Nothing
        End If
        ' Disables "read" timer.
        timerRead.Enabled = False
        ' Updates enable state of manual read button.
        buttonMsgRead.Enabled = isConnected() AndAlso radioButtonMsgManual.Checked
        ' Updates timer that diplays new messages.
        timerDisplay.Enabled = isConnected()
        ' Aborts function if no channel is connected.
        If Not isConnected() Then
            Return
        End If
        ' According to the kind of reading, either a timer, a thread or a button is enabled.
        If radioButtonMsgTimer.Checked Then
            ' Enables Timer.
            timerRead.Enabled = isConnected()
        ElseIf radioButtonMsgEvent.Checked Then
            'Creates and starts the tread to read CAN ISO-TP Message using PCANTP_PARAM_RECEIVE_EVENT.
            Dim threadDelegate As New System.Threading.ThreadStart(AddressOf pctpReadThread)
            m_readThread = New System.Threading.Thread(threadDelegate)
            m_readThread.IsBackground = True
            m_readThread.Start()
        End If
    End Sub

    ''' <summary>
    ''' Event handler called periodically to update the display of received messages.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub timerDisplay_Tick(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles timerDisplay.Tick
        displayMessages()
    End Sub
    ''' <summary>
    ''' Event handler called periodically to read new ISO-TP messages.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub timerRead_Tick(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles timerRead.Tick
        ' Try to read new messages.
        readMessages()
    End Sub
#End Region

#Region "Messages > Write Event Handlers"
    ''' <summary>
    ''' Event handler called when the button "Fill message's data" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonMsgDataFill_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonMsgDataFill.Click
        Dim buf As StringBuilder
        Dim byteStr As Char()

        ' Generates a 4095-bytes String (each byte is increased by 1)
        buf = New StringBuilder(4095 * 2)
        Dim i As Integer = 1, count As Integer = 0
        While count < buf.Capacity
            byteStr = [String].Format("{0:X2}", i Mod 256, 16).ToCharArray()
            buf.Append(byteStr(0))
            buf.Append(byteStr(1))
            i += 1
            count += 2
        End While
        ' Sets the data according to its length
        textBoxMsgData.Text = buf.ToString().Substring(0, textBoxMsgData.MaxLength).ToUpper()
    End Sub
    ''' <summary>
    ''' Event handler called when the button "Write message" is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonMsgWrite_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonMsgWrite.Click
        Dim msg As TPCANTPMsg
        Dim mapping As MappingStatus
        Dim sts As TPCANTPStatus
        Dim strByte As String

        ' Asserts a mapping to write is selected.
        If (comboBoxMsgMapping.SelectedItem Is Nothing) Then
            Return
        End If
        ' Gets the selected mapping and asserts it is valid.
        mapping = DirectCast(DirectCast(comboBoxMsgMapping.SelectedItem, ComboBoxItem).Data, MappingStatus)
        If mapping Is Nothing Then
            Return
        End If
        ' Initializes the TPCANTPMsg to send with the selected mapping.
        msg.IDTYPE = mapping.m_canIdType
        msg.FORMAT = mapping.m_formatType
        msg.MSGTYPE = mapping.m_msgType
        msg.TA_TYPE = mapping.m_targetType
        If mapping.IsAutomatic Then
            ' Be cautious with Enhanced format addressing which uses 11 bits for addresses
            If (mapping.m_formatType <> TPCANTPFormatType.PCANTP_FORMAT_ENHANCED) Then
                msg.SA = Convert.ToByte(numericUpDownSourceAddr.Value)
                msg.TA = Convert.ToByte(numericUpDownTargetAddr.Value)
                msg.RA = Convert.ToByte(numericUpDownRemoteAddr.Value)
            Else
                ' Enhanced format addressing uses 11 bits for its addresses,
                '  in this case Remote address is used to store the extra bits
                '  from Source and Target addresses.
                Dim iBuf As UInteger
                ' Unique SA is 11bits: store bits [7..0] in SA
                ' and the rest in [7..4] of RA
                iBuf = Convert.ToUInt32(numericUpDownSourceAddr.Value)
                msg.SA = CByte(iBuf)
                msg.RA = CByte((iBuf >> 8) << 4)
                ' Unique TA is 11bits: store bits [7..0] in TA
                ' and the rest in [3..0] of RA
                iBuf = Convert.ToUInt32(numericUpDownTargetAddr.Value)
                msg.TA = CByte(iBuf)
                msg.RA = CByte(iBuf >> 8)
            End If
        Else
            ' The mapping defines the addresses.
            msg.SA = mapping.m_sourceAddr
            msg.TA = mapping.m_targetAddr
            msg.RA = mapping.m_remoteAddr
        End If
        ' Initializes RESULT now to avoid a strange behaviour within Visual Studio while debugging (C++.NET issue).
        msg.RESULT = TPCANTPConfirmation.PCANTP_N_OK
        ' Sets length and data.
        msg.LEN = Convert.ToUInt16(numericUpDownMsgLength.Value)
        msg.DATA = New Byte(4094) {}
        For i As Integer = 0 To msg.LEN - 1
            If (i * 2 < textBoxMsgData.TextLength - 1) Then
                strByte = textBoxMsgData.Text.Substring(i * 2, 2)
            ElseIf (i * 2 < textBoxMsgData.TextLength) Then
                strByte = textBoxMsgData.Text.Substring(i * 2, 1)
            Else
                strByte = "0"
            End If
            msg.DATA(i) = Convert.ToByte(strByte, 16)
        Next
        ' CAN FD support
        If (checkBoxFDMessage.Checked) Then
            msg.IDTYPE = msg.IDTYPE Or TPCANTPIdType.PCANTP_ID_CAN_FD
            ' if BRS support
            If (checkBoxBRS.Checked) Then
                msg.IDTYPE = msg.IDTYPE Or TPCANTPIdType.PCANTP_ID_CAN_BRS
            End If
        End If
        ' Manage priority (available only in 29 bits mixed of fixed normal)
        If (checkBoxHasPriority.Checked) Then
            If ((mapping.m_formatType = TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL) Or (mapping.m_formatType = TPCANTPFormatType.PCANTP_FORMAT_MIXED) Or
                (mapping.m_formatType = TPCANTPFormatType.PCANTP_FORMAT_ENHANCED)) Then
                If (TPCANTPIdTypePriority.PCANTP_ID_CAN_IS_EXTENDED(msg.IDTYPE)) Then
                    msg.IDTYPE = msg.IDTYPE Or CanTpApi.PCANTP_ID_CAN_IS_PRIORITY_MASK
                    ' Clean old priority
                    msg.IDTYPE = msg.IDTYPE And CanTpApi.PCANTP_ID_CAN_MASK + CanTpApi.PCANTP_ID_CAN_IS_PRIORITY_MASK
                    ' Add priority value
                    msg.IDTYPE = msg.IDTYPE Or (numericUpDownPriority.Value << 5)
                End If
            End If
        End If
        ' Sends the message to the Tx queue of the API.
        sts = CanTpApi.Write(m_pctpHandle, msg)
        checkCanTpStatus(m_pctpHandle, sts, msg)
    End Sub

    ''' <summary>
    ''' Event handler called when selection of comboBox "Write mapping" is changed.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub comboBoxMsgMapping_SelectedIndexChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles comboBoxMsgMapping.SelectedIndexChanged
        Dim mapping As MappingStatus
        Dim enabled As Boolean

        ' Asserts a mapping is selected.
        mapping = Nothing
        If comboBoxMsgMapping.SelectedItem IsNot Nothing Then
            ' Gets the selected mapping.
            mapping = DirectCast(DirectCast(comboBoxMsgMapping.SelectedItem, ComboBoxItem).Data, MappingStatus)
            ' Enables the button to write the message.
            buttonMsgWrite.Enabled = True
            ' Source (SA), Target (TA) and Remote address (RA) textboxes are 
            ' enabled ONLY if the mapping allows dynamic 
            ' generation of CAN ID based on SA, TA, RA.
            ' i.e. only if an "automatic mapping" is selected.
            enabled = mapping.IsAutomatic
            numericUpDownSourceAddr.Enabled = enabled
            numericUpDownTargetAddr.Enabled = enabled
            ' RA is also only available with Remote Diagnostic Messages.
            If mapping.m_msgType = TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC Then
                numericUpDownRemoteAddr.Enabled = enabled
            Else
                numericUpDownRemoteAddr.Enabled = False
                ' To avoid confusion, RA text is reset to 0x00.
                numericUpDownRemoteAddr.Text = "0"
            End If
            ' Sets SA/TA maximum values (default is 8bits, but Enhanced addressing uses 11bits for SA and TA
            numericUpDownSourceAddr.Maximum = If((mapping.m_formatType = TPCANTPFormatType.PCANTP_FORMAT_ENHANCED), &H7FF, &HFF)
            numericUpDownTargetAddr.Maximum = numericUpDownSourceAddr.Maximum
            ' Sets SA/TA/RA as defined in the mapping.
            ' If an "automatic mapping" is selected,
            ' then the user has to set those value.
            If Not mapping.IsAutomatic Then
                numericUpDownSourceAddr.Value = mapping.m_sourceAddr
                numericUpDownTargetAddr.Value = mapping.m_targetAddr
                numericUpDownRemoteAddr.Value = mapping.m_remoteAddr
            End If
            ' Sets the maximum size of the ISO-TP message.
            If mapping.m_targetType = TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL Then

                ' Functional addressing allows only Single Frame to be sent
                numericUpDownMsgLength.Maximum = CanTpUtils.ISOTP_MSG_FUNC_MAX_LENGTH
                If (mapping.m_formatType <> TPCANTPFormatType.PCANTP_FORMAT_NORMAL AndAlso
                    mapping.m_formatType <> TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL) Then
                    numericUpDownMsgLength.Maximum -= 1
                End If
            Else
                numericUpDownMsgLength.Maximum = CanTpUtils.ISOTP_MSG_PHYS_MAX_LENGTH
            End If
            If (((mapping.m_formatType = TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL) Or (mapping.m_formatType = TPCANTPFormatType.PCANTP_FORMAT_MIXED) Or
                (mapping.m_formatType = TPCANTPFormatType.PCANTP_FORMAT_ENHANCED)) And mapping.m_canIdType = TPCANTPIdType.PCANTP_ID_CAN_29BIT) Then
                ' If it Is a 29bits mapping fixed normal, enhanced or mixed, changing "Priority" value Is allowed
                checkBoxHasPriority.Enabled = True
            Else
                checkBoxHasPriority.Enabled = False
                checkBoxHasPriority.Checked = False
                numericUpDownPriority.Enabled = False
            End If
            ' No mapping is selected in comboBox.
        Else
            ' Disables "write message" UI components.
            enabled = False
            buttonMsgWrite.Enabled = enabled
            numericUpDownSourceAddr.Enabled = enabled
            numericUpDownTargetAddr.Enabled = enabled
            numericUpDownRemoteAddr.Enabled = enabled
        End If
    End Sub

    ''' <summary>
    ''' Event handler called when numericUpDown of "Message's length" is changed,
    ''' or textBoxMsgData is left.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub numericUpDownMsgLength_ValueChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles numericUpDownMsgLength.ValueChanged
        ' Gets the size of the text based on the data length.
        ' Note: each byte is translated in 2 characters.
        textBoxMsgData.MaxLength = Convert.ToInt32(numericUpDownMsgLength.Value) * 2
        ' Adds a zero to the last "non-two-character" byte (ex. 0xF => 0x0F ; 0xFA1 => 0xFA01)
        If textBoxMsgData.Text.Length Mod 2 = 1 Then
            textBoxMsgData.Text = textBoxMsgData.Text.Insert(textBoxMsgData.Text.Length - 1, "0")
        End If
        ' Modify data string based on the new length
        If (textBoxMsgData.Text.Length > textBoxMsgData.MaxLength) Then
            ' Data is longer than the length, truncates the string
            textBoxMsgData.Text = textBoxMsgData.Text.Substring(0, textBoxMsgData.MaxLength)
        Else
            ' Data is shroter than the length, pads the string with zeros
            textBoxMsgData.Text = textBoxMsgData.Text.PadRight(textBoxMsgData.MaxLength, "0"c)
        End If
    End Sub

    ''' <summary>
    ''' Event handler called when a key is pressed inside "Write message Data" textBox.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub textBoxHexOnly_KeyPress(ByVal sender As System.Object, ByVal e As System.Windows.Forms.KeyPressEventArgs) Handles textBoxMsgData.KeyPress
        Dim chCheck As Char

        ' Converts the Character to its Upper case equivalent
        chCheck = Char.ToUpper(e.KeyChar)
        ' Key is the Delete (Backspace) Key
        If Asc(chCheck) = 8 Then
            Return
        End If
        e.KeyChar = chCheck
        ' Key is a number between 0-9
        If (Asc(chCheck) > 47) AndAlso (Asc(chCheck) < 58) Then
            Return
        End If
        ' Key is a character between A-F
        If (Asc(chCheck) > 64) AndAlso (Asc(chCheck) < 71) Then
            Return
        End If
        ' key is neither a number nor a character between A(a) and F(f)
        e.Handled = True
    End Sub
#End Region

#Region "Controllers handling ISO-TP connection"
    ''' <summary>
    ''' Connects to a PCAN ISO-TP channel and sets m_pctpHandle.
    ''' </summary>
    ''' <param name="canHandle">A PCANTP Channel Handle representing a PCANTP-Client</param>
    ''' <param name="baudrate">The CAN Hardware speed</param>
    ''' <param name="hwType">Non plug-'n-play: The type of hardware and operation mode</param>
    ''' <param name="ioPort">Non plug-'n-play: The I/O address for the parallel port</param>
    ''' <param name="interrupt">Non plug-'n-play: Interrupt number of the parallel port</param>
    ''' <returns>true is returned on success</returns>
    Private Function connect(ByVal canHandle As TPCANTPHandle, ByVal baudrate As TPCANTPBaudrate, ByVal hwType As TPCANTPHWType, ByVal ioPort As UInteger, ByVal interrupt As Byte) As Boolean
        Dim sts As TPCANTPStatus

        ' Connects to the selected PCAN-ISO-TP channel.
        ' If FD support
        If (checkBoxCanFd.Checked) Then
            ' Convert unicode string to multibytes char
            sts = CanTpApi.InitializeFD(canHandle, textBoxCanFdBitrate.Text)
            checkCanTpStatus(canHandle, sts, Encoding.ASCII.GetBytes(textBoxCanFdBitrate.Text.ToString()))
        Else
            sts = CanTpApi.Initialize(canHandle, baudrate, hwType, ioPort, interrupt)
            checkCanTpStatus(canHandle, sts, baudrate, hwType, ioPort, interrupt)
        End If
        If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
            ' Stores the currently connected channel.
            m_pctpHandle = canHandle
            ' Clear mappings.
            mappingsInit()
            mappingsChanged()
            ' Desable FD checkbox
            checkBoxCanFd.Enabled = False
        Else
            m_pctpHandle = CanTpApi.PCANTP_NONEBUS
        End If
        ' Sets the connection status of the main-form.
        setConnectionStatus(sts = TPCANTPStatus.PCANTP_ERROR_OK)
        Return (m_pctpHandle <> CanTpApi.PCANTP_NONEBUS)
    End Function
    ''' <summary>
    ''' Disconnects the currently connected channel (m_pctpHandle)
    ''' </summary>
    ''' <returns>true is returned on success</returns>
    Private Function disconnect() As Boolean
        Dim sts As TPCANTPStatus

        ' Disconnects from selected PCAN-ISO-TP channel
        sts = CanTpApi.Uninitialize(m_pctpHandle)
        checkCanTpStatus(m_pctpHandle, sts)
        If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
            ' Resets channel and stored mappings.
            m_pctpHandle = CanTpApi.PCANTP_NONEBUS
            ' Resets stored mappings.
            m_mappings.Clear()
            mappingsChanged()
            numericUpDownPriority.Enabled = False
            checkBoxHasPriority.Checked = False
            checkBoxHasPriority.Enabled = False
            ' enable FD checkbox
            checkBoxCanFd.Enabled = True
        End If
        ' Sets the connection status of the main-form
        setConnectionStatus(sts <> TPCANTPStatus.PCANTP_ERROR_OK)
        Return isConnected()
    End Function
#End Region

#Region "Controllers handling ISO-TP mappings"
    ''' <summary>
    ''' Sets the default "automatic" mappings.
    ''' Those mappings are set for tutorial/understanding purpose,
    ''' ISO-TP API handles internally CANTP messages with those
    ''' network information and do not require mappings to be defined.
    ''' </summary>
    Private Sub mappingsInit()
        Dim mapping As MappingStatus

        ' Initializes the internal list of mappings
        If m_mappings Is Nothing Then
            m_mappings = New List(Of MappingStatus)()
        Else
            m_mappings.Clear()
        End If
        ' 29 bits CAN ID, Normal Fixed and physical addressing
        mapping = New MappingStatus(0, 0, TPCANTPIdType.PCANTP_ID_CAN_29BIT,
            TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL, TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC,
            TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL, 0, 0, 0)
        m_mappings.Add(mapping)
        ' 29 bits CAN ID, Normal Fixed and functional addressing
        mapping = New MappingStatus(0, 0, TPCANTPIdType.PCANTP_ID_CAN_29BIT,
            TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL, TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC,
            TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL, 0, 0, 0)
        m_mappings.Add(mapping)
        ' 29 bits CAN ID, Mixed and physical addressing
        mapping = New MappingStatus(0, 0, TPCANTPIdType.PCANTP_ID_CAN_29BIT,
            TPCANTPFormatType.PCANTP_FORMAT_MIXED, TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
            TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL, 0, 0, 0)
        m_mappings.Add(mapping)
        ' 29 bits CAN ID, Mixed and functional addressing
        mapping = New MappingStatus(0, 0, TPCANTPIdType.PCANTP_ID_CAN_29BIT,
            TPCANTPFormatType.PCANTP_FORMAT_MIXED, TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
            TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL, 0, 0, 0)
        m_mappings.Add(mapping)
        ' 29 bits CAN ID, Enhanced and physical addressing (ISO 15765-3)
        mapping = New MappingStatus(0, 0, TPCANTPIdType.PCANTP_ID_CAN_29BIT,
                TPCANTPFormatType.PCANTP_FORMAT_ENHANCED, TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC,
                TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL, 0, 0, 0)
        m_mappings.Add(mapping)
    End Sub
    ''' <summary>
    ''' Sets a new mapping to the ISO-TP API.
    ''' </summary>
    ''' <param name="mapping"></param>
    Private Sub mappingsAdd(ByVal mapping As MappingStatus)
        Dim sts As TPCANTPStatus

        ' Adds the mapping inside the API
        sts = CanTpApi.AddMapping(m_pctpHandle, mapping.m_canId, mapping.m_canIdResponse,
               mapping.m_canIdType, mapping.m_formatType, mapping.m_msgType,
               mapping.m_sourceAddr, mapping.m_targetAddr, mapping.m_targetType, mapping.m_remoteAddr)
        checkCanTpStatus(m_pctpHandle, sts, mapping)
        If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
            ' Adds the mapping to the internal list of configured mappings.
            m_mappings.Add(mapping)
        End If
    End Sub
    ''' <summary>
    ''' Function to be called when a change occured in the configured mappings.
    ''' The function updates all UI components according to the new settings.
    ''' </summary>
    Private Sub mappingsChanged()
        Dim mappingComboSel As MappingStatus    ' selected mapping in comboBoxMsgMapping
        Dim mappingListSel As MappingStatus     ' selected mapping in listViewMappings
        Dim mapping As MappingStatus
        Dim lvi As ListViewItem

        mappingComboSel = Nothing
        mappingListSel = Nothing
        ' Store selected item in mappingListSel.
        If listViewMappings.SelectedItems.Count > 0 Then
            mappingListSel = DirectCast(listViewMappings.SelectedItems(0).Tag, MappingStatus)
        End If
        ' Store selected item in comboBoxMsgMapping.
        If (comboBoxMsgMapping.SelectedItem IsNot Nothing) Then
            mappingComboSel = DirectCast(DirectCast(comboBoxMsgMapping.SelectedItem, ComboBoxItem).Data, MappingStatus)
        End If
        ' Clear listview of mappings.
        listViewMappings.Items.Clear()
        ' Clear comboBox of mappings in "write message" section.
        comboBoxMsgMapping.Items.Clear()
        ' Loop through all configured mappings and add them to UI.
        For i As Integer = 0 To m_mappings.Count - 1
            mapping = m_mappings(i)
            ' Add mapping to listView of mappings.
            lvi = New ListViewItem()
            lvi.Tag = mapping
            lvi.Text = mapping.CanId
            lvi.SubItems.Add(mapping.CanIdResponse)
            lvi.SubItems.Add(mapping.TargetType)
            lvi.SubItems.Add(mapping.CanIdType)
            lvi.SubItems.Add(mapping.SourceAddress)
            lvi.SubItems.Add(mapping.TargetAddress)
            lvi.SubItems.Add(mapping.MsgType)
            lvi.SubItems.Add(mapping.FormatType)
            lvi.SubItems.Add(mapping.RemoteAddress)
            listViewMappings.Items.Add(lvi)
            If mapping.Equals(mappingComboSel) Then
                lvi.Selected = True
            End If
            ' Add mapping to comboBox of mappings in the messages/write tab.
            comboBoxMsgMapping.Items.Add(New ComboBoxItem(mapping.Name, mapping))
            If (mapping.Equals(mappingComboSel)) Then
                comboBoxMsgMapping.SelectedIndex = i
            End If
        Next
    End Sub
#End Region

#Region "Controllers handling ISO-TP messages"
    ''' <summary>
    ''' Reads several ISO-TP messages from the bus CAN.
    ''' </summary>
    Private Sub readMessages()
        Dim sts As TPCANTPStatus

        ' We read at least one time the queue looking for messages.
        ' If a message is found, we look again trying to find more.
        ' If the queue is empty or an error occurr, we get out from
        ' the dowhile statement.
        Do
            sts = readMessage()
        Loop While (isConnected() AndAlso sts = TPCANTPStatus.PCANTP_ERROR_OK)
    End Sub
    ''' <summary>
    ''' Reads an ISO-TP messages from the bus CAN.
    ''' </summary>
    ''' <returns>A TPCANTPStatus error code</returns>
    Private Function readMessage() As TPCANTPStatus
        Dim msg As TPCANTPMsg = Nothing
        Dim ts As TPCANTPTimestamp
        Dim sts As TPCANTPStatus

        ' Reads and process a single ISO-TP message
        sts = CanTpApi.Read(m_pctpHandle, msg, ts)
        If sts = TPCANTPStatus.PCANTP_ERROR_OK Then
            processMessage(msg, ts)
            checkCanTpStatus(m_pctpHandle, sts, msg)
        End If
        Return sts
    End Function
    ''' <summary>
    ''' Processes a received message, in order to show it in the Message-ListView
    ''' </summary>
    ''' <param name="msgRcvd">The received ISO-TP message</param>
    Private Sub processMessage(ByVal msgRcvd As TPCANTPMsg, ByVal ts As TPCANTPTimestamp)
        ' Prevent concurrent access (reading thread and UI event can alter the list)
        SyncLock m_receiveMsgs.SyncRoot
            ' Searches if the message has already been received 
            '  (same Network Address Information) or if it is a new message.
            For Each msg As MessageStatus In m_receiveMsgs
                If (msg.isSameNetAddrInfo(msgRcvd)) Then
                    ' Modifies the existing message and exit
                    msg.update(msgRcvd, ts)
                    Return
                End If
            Next
            ' Message not found, it will be created
            insertMsgEntry(msgRcvd, ts)
        End SyncLock
    End Sub
    ''' <summary>
    ''' Inserts a new entry for a new message in the Message-ListView
    ''' </summary>
    ''' <param name="msg">The messasge to be inserted</param>
    ''' <param name="ts">The Timesamp of the new message</param>
    Private Sub insertMsgEntry(ByVal msg As TPCANTPMsg, ByVal ts As TPCANTPTimestamp)
        Dim msgSts As MessageStatus   ' MessageStatus corresponding to the new TPCANTPMsg
        Dim lvi As ListViewItem       ' ListView Item corresponding to the MessageStatus
        Dim lviSub As ListViewItem.ListViewSubItem       ' SubItem of the listView Item

        SyncLock (m_receiveMsgs.SyncRoot)
            ' Adds a new MessageStatus in the received-message list.
            msgSts = New MessageStatus(msg, ts, listViewMsgs.Items.Count)
            msgSts.ShowingPeriod = checkBoxMsgShowPeriod.Checked
            m_receiveMsgs.Add(msgSts)
            ' Adds a corresponding new item in the message listView
            lvi = listViewMsgs.Items.Add(msgSts.SourceAddress)
            lvi.UseItemStyleForSubItems = False
            lvi.SubItems.Add(msgSts.TargetAddress)
            lvi.SubItems.Add(msgSts.RemoteAddress)
            lvi.SubItems.Add(msgSts.CanIdType)
            lvi.SubItems.Add(msgSts.MsgType)
            lvi.SubItems.Add(msgSts.FormatType)
            lvi.SubItems.Add(msgSts.TargetType)
            lviSub = lvi.SubItems.Add(msgSts.ResultString)
            ' Highlights network ISO-TP error in red
            If msgSts.CanTpMsg.RESULT <> TPCANTPConfirmation.PCANTP_N_OK Then
                lviSub.ForeColor = Color.Red
            End If
            lvi.SubItems.Add(msgSts.LengthString)
            lvi.SubItems.Add(msgSts.Count.ToString())
            lvi.SubItems.Add(msgSts.TimeString)
            lvi.SubItems.Add(msgSts.DataString)
        End SyncLock
    End Sub
    ''' <summary>
    ''' Updates the "received-message" listView with the internal list of received message.
    ''' </summary>
    Private Sub displayMessages()
        Dim lvi As ListViewItem  ' listView item corresponding to a specific messageStatus

        SyncLock m_receiveMsgs.SyncRoot
            ' Loops through all the received messageStatus to update the display
            For Each msgStatus As MessageStatus In m_receiveMsgs
                ' Checks that the data needs to be actualized
                If msgStatus.MarkedAsUpdated Then
                    msgStatus.MarkedAsUpdated = False
                    ' Retrieves the listView corresponding to the messageStatus
                    lvi = listViewMsgs.Items(msgStatus.Position)
                    ' Updates the listView item's fields
                    lvi.SubItems(0).Text = msgStatus.SourceAddress
                    lvi.SubItems(1).Text = msgStatus.TargetAddress
                    lvi.SubItems(2).Text = msgStatus.RemoteAddress
                    lvi.SubItems(3).Text = msgStatus.CanIdType
                    lvi.SubItems(4).Text = msgStatus.MsgType
                    lvi.SubItems(5).Text = msgStatus.FormatType
                    lvi.SubItems(6).Text = msgStatus.TargetType
                    lvi.SubItems(7).Text = msgStatus.ResultString
                    ' Updates the Network Result color (red for an error)
                    If msgStatus.CanTpMsg.RESULT <> TPCANTPConfirmation.PCANTP_N_OK Then
                        lvi.SubItems(7).ForeColor = Color.Red
                    Else
                        lvi.SubItems(7).ForeColor = lvi.ForeColor
                    End If
                    lvi.SubItems(8).Text = msgStatus.LengthString
                    lvi.SubItems(9).Text = msgStatus.Count.ToString()
                    lvi.SubItems(10).Text = msgStatus.TimeString
                    lvi.SubItems(11).Text = msgStatus.DataString
                End If
            Next
        End SyncLock
    End Sub
#End Region

#Region "Controllers handling thread to read message"
    ''' <summary>
    ''' Main entry point for thread "read messages"
    ''' </summary>
    Private Sub pctpReadThread()
        Dim iBuffer As UInt32
        Dim sts As TPCANTPStatus

        ' Sets the handle of the Receive-Event.
        iBuffer = Convert.ToUInt32(m_receiveEvent.SafeWaitHandle.DangerousGetHandle().ToInt32())
        sts = CanTpApi.SetValue(m_pctpHandle, TPCANTPParameter.PCANTP_PARAM_RECEIVE_EVENT, iBuffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuffer), UInteger))
        If sts <> TPCANTPStatus.PCANTP_ERROR_OK Then
            ' Handles failure on the UI thread then aborts thread.
            Me.Invoke(m_readFailedDelegate, sts)
            Return
        End If
        ' Loops while the "read thread" mode is selected
        While radioButtonMsgEvent.Checked
            ' Waits for Receive-Event
            If (m_receiveEvent.WaitOne(50)) Then
                ' Process Receive-Event using .NET Invoke function
                ' in order to interact with Winforms UI (calling the 
                ' function readMessages)
                Me.Invoke(m_readDelegate)
            End If
        End While
    End Sub
    ''' <summary>
    ''' Handles failure while reading messages with a thread.
    ''' </summary>
    ''' <param name="sts"></param>
    Private Sub readThreadFailed(ByVal sts As TPCANTPStatus)
        ' Shows the error status.
        checkCanTpStatus(sts)
        ' reverts to the default reading method.
        radioButtonMsgTimer.PerformClick()
    End Sub
#End Region

#Region "Controllers handling miscellaneous UI updates."
    ''' <summary>
    ''' Checks a TPCANTPStatus value and shows a messageBox if the status is not OK.
    ''' </summary>
    ''' <param name="err">ISO-TP Status to check</param>
    Private Sub checkCanTpStatus(ByVal err As TPCANTPStatus)
        If err = TPCANTPStatus.PCANTP_ERROR_OK Then
            Return
        End If
        MessageBox.Show(Me, CanTpUtils.GetError(err) + " (" + err.ToString + ")",
            "CANTP Status", MessageBoxButtons.OK, MessageBoxIcon.Exclamation)
    End Sub


    ''' <summary>
    ''' Check the status of CAN function result And give information to compare results in debug mode
    ''' </summary>
    ''' <param name="p_Handle">CAN Handle</param>
    ''' <param name="p_Err">CAN Status</param>
    ''' <param name="p_Args">List of arguments given to CANTP function (usage : {(int)var1, (int) var2, ... } </param>
    ''' <param name="p_Msg">Pointer on CAN Message structure</param>
    Private Sub checkCanTpStatus(ByVal p_Handle As TPCANTPHandle, ByVal p_Err As TPCANTPStatus, ByVal p_Msg As TPCANTPMsg, ParamArray p_Args() As UInteger)
        Dim strOutMsg As String = ""
        Dim strCallerName As String = ""
        Dim strData As String = ""

        Dim polynomialIndex As UInteger = 0
        Const maxsize As UInteger = 256
        Dim length As UShort = System.Convert.ToUInt16(p_Args.Length)
        Dim datatab(maxsize) As Byte
        Dim result As UInteger = 0
        Dim counter As UInteger = 0

#If _DEBUG_WRITE_CHECK_FILE Then
        Dim checkFileName = ""
        If (g_FileInitAlreadyDone = False) Then
            ' Init file to write result
            Dim sTime As String = DateTime.Now.ToString("yyyy.MM.dd_HH.mm.ss")
            checkFileName = "VB-" + sTime + ".csv"
            Try
                g_CheckFileToWrite = New System.IO.StreamWriter(checkFileName, False)

                g_CheckFileToWrite.WriteLine("CallerName;Result;Args_1;Args_2;Args_3;Args_4;Args_5;Args_6;Args_7;Args_8;Args_9;Args_10;Args_11")
                g_CheckFileToWrite.Flush()
                g_FileInitAlreadyDone = True
            Catch ex As Exception

            End Try
        End If
#End If

        ' Get caller method name
        strCallerName = GetCallerMethodName()

        '  Polynomial calculation for unique result from Args
        polynomialIndex += 1
        result += (polynomialIndex) * p_Handle
        polynomialIndex += 1
        result += (polynomialIndex) * System.Convert.ToUInt32(p_Err)

        strData = String.Format("Handle[{0}];Status[{1}]", p_Handle, System.Convert.ToUInt32(p_Err))
        For i As Integer = 0 To p_Args.Length - 1
            datatab(polynomialIndex) = System.Convert.ToByte(&HFF And (p_Args(i) * (polynomialIndex + 1)))
            polynomialIndex += 1
            Dim calcUInt1 As UInt64 = System.Convert.ToUInt64(UInteger.MaxValue And (p_Args(i) * (polynomialIndex + 1)))
            Dim calcUInt2 As UInt64 = System.Convert.ToUInt64(result) + System.Convert.ToUInt64(calcUInt1)
            result = UInteger.MaxValue And calcUInt2
            strData = String.Format("{0};Arg_{1}[{2}]", strData, counter, System.Convert.ToUInt32(p_Args(i)))
            counter += 1
        Next i


        ' If there Is a message structure a CRC Is calculated dans message Is added in out string
        If (p_Msg.LEN > 0) Then

            polynomialIndex += 1
            result += (polynomialIndex) * p_Msg.SA
            polynomialIndex += 1
            result += (polynomialIndex) * p_Msg.TA
            polynomialIndex += 1
            result += (polynomialIndex) * System.Convert.ToUInt32(p_Msg.TA_TYPE)
            polynomialIndex += 1
            result += (polynomialIndex) * p_Msg.RA
            polynomialIndex += 1
            result += (polynomialIndex) * System.Convert.ToUInt32(p_Msg.IDTYPE)
            polynomialIndex += 1
            result += (polynomialIndex) * System.Convert.ToUInt32(p_Msg.MSGTYPE)
            polynomialIndex += 1
            result += (polynomialIndex) * System.Convert.ToUInt32(p_Msg.FORMAT)
            polynomialIndex += 1
            result += (polynomialIndex) * System.Convert.ToUInt32(p_Msg.RESULT)
            polynomialIndex += 1
            result += (polynomialIndex) * p_Msg.LEN

            ' All Data fields are used in case of bad data indexing during message construction
            ' for (int i = 0; i < 4095; i++) retVal += (++p_polynomeIndex) * p_canMsg.DATA[i];
            ' A CRC 16 Is calculated on DATA to prevent LONG64 overflow with polynome on 4095 fields
            ' And Not a CRC32 for perfomances improvement
            result += crc16(p_Msg.DATA, p_Msg.LEN)

            strData = String.Format("{0};SA[{1}];TA[{2}];TA_TYPE[{3}];RA[{4}];IDTYPE[{5}];MSGTYPE[{6}];FORMAT[{7}];RESULT[{8}];LEN[{9}];DATA[",
                    strData, p_Msg.SA, p_Msg.TA, System.Convert.ToUInt32(p_Msg.TA_TYPE), p_Msg.RA, System.Convert.ToUInt32(p_Msg.IDTYPE), (p_Msg.MSGTYPE),
                                    System.Convert.ToUInt32(p_Msg.FORMAT), System.Convert.ToUInt32(p_Msg.RESULT), p_Msg.LEN)
            For i As Integer = 0 To p_Msg.LEN

                strData = String.Format("{0}{1}", strData, p_Msg.DATA(i).ToString("X2"))

            Next i
            strData = String.Format("{0}]", strData)
        End If

        ' Concat all strings
        strOutMsg = String.Format("{0};{1};{2}", strCallerName, result, strData)
#If _DEBUG_WRITE_CHECK_FILE Then
#If _DEBUG Then
        Console.WriteLine(strOutMsg)
#End If
        If (g_FileInitAlreadyDone = True) Then

            ' Write result in file
            Try
                g_CheckFileToWrite.WriteLine(strOutMsg)
                g_CheckFileToWrite.Flush()

            Catch ex As Exception
            End Try
        End If
#End If

        If (p_Err = TPCANTPStatus.PCANTP_ERROR_OK) Then
            Return
        End If
        MessageBox.Show(CanTpUtils.GetError(p_Err) + " (" + p_Err.ToString + ")",
            "CANTP Status", MessageBoxButtons.OK, MessageBoxIcon.Exclamation)
    End Sub

    ''' <summary>
    ''' Check the status of CAN function result and give information to compare results in debug mode
    ''' </summary>
    ''' <param name="p_Handle">CAN Handle</param>
    ''' <param name="p_Err">CAN Status</param>
    ''' <param name="p_Args">List of arguments given to CANTP function (usage : {(int)var1, (int) var2, ... } </param>
    Private Sub checkCanTpStatus(p_Handle As TPCANTPHandle, p_Err As TPCANTPStatus, ParamArray p_Args() As UInteger)
        Dim p_Msg As TPCANTPMsg
        p_Msg.RA = 0
        p_Msg.SA = 0
        p_Msg.TA = 0
        p_Msg.LEN = 0
        p_Msg.DATA = New Byte(4095) {}
        p_Msg.FORMAT = TPCANTPFormatType.PCANTP_FORMAT_UNKNOWN
        p_Msg.IDTYPE = TPCANTPIdType.PCANTP_ID_CAN_11BIT
        p_Msg.MSGTYPE = TPCANTPMessageType.PCANTP_MESSAGE_UNKNOWN
        p_Msg.RESULT = TPCANTPConfirmation.PCANTP_N_ERROR
        p_Msg.TA_TYPE = TPCANTPAddressingType.PCANTP_ADDRESSING_UNKNOWN

        checkCanTpStatus(p_Handle, p_Err, p_Msg, p_Args)
    End Sub
    ''' <summary>
    ''' Check the status of CAN function result and give information to compare results in debug mode
    ''' </summary>
    ''' <param name="p_Handle">CAN Handle</param>
    ''' <param name="p_Err">CAN Status</param>
    ''' <param name="p_Msg">Characters buffer</param>
    ''' <param name="p_Parameter">Parameter</param>
    Private Sub checkCanTpStatus(p_Handle As TPCANTPHandle, p_Err As TPCANTPStatus, p_Msg() As Byte, Optional p_Parameter As Byte = 0)
        If (p_Msg.Length <> 0) Then
            checkCanTpStatus(p_Handle, p_Err, p_Parameter, crc16(p_Msg, p_Msg.Length))
        Else
            checkCanTpStatus(p_Handle, p_Err)
        End If
    End Sub
    ''' <summary>
    ''' Check the status of CAN function result and give information to compare results in debug mode
    ''' </summary>
    ''' <param name="p_Handle">CAN Handle</param>
    ''' <param name="p_Err">CAN Status</param>
    ''' <param name="p_Msg">Characters buffer</param>
    ''' <param name="p_Parameter">Parameter</param>
    Private Sub checkCanTpStatus(p_Handle As TPCANTPHandle, p_Err As TPCANTPStatus, p_Msg As StringBuilder, Optional p_Parameter As TPCANTPParameter = 0)
        If (p_Msg.Length <> 0) Then
            checkCanTpStatus(p_Handle, p_Err, p_Parameter, crc16(Encoding.ASCII.GetBytes(p_Msg.ToString()), p_Msg.Length))
        Else
            checkCanTpStatus(p_Handle, p_Err)
        End If
    End Sub
    ''' <summary>
    ''' Check the status of CAN function result and give information to compare results in debug mode
    ''' </summary>
    ''' <param name="p_Handle">CAN Handle</param>
    ''' <param name="p_Err">CAN Status</param>
    ''' <param name="p_Mapping">Pointer on mapping structure</param>
    Private Sub checkCanTpStatus(ByVal p_Handle As TPCANTPHandle, ByVal p_Err As TPCANTPStatus, ByVal p_Mapping As MappingStatus)
        checkCanTpStatus(p_Handle, p_Err,
                         p_Mapping.m_canId, p_Mapping.m_canIdResponse,
                         p_Mapping.m_canIdType, p_Mapping.m_formatType, p_Mapping.m_msgType,
                         p_Mapping.m_sourceAddr, p_Mapping.m_targetAddr, p_Mapping.m_targetType, p_Mapping.m_remoteAddr)
    End Sub
    ''' <summary>
    ''' Get caller name
    ''' </summary>
    ''' <returns>Return the name of the caller</returns>
    Private Function GetCallerMethodName() As String

        Dim frame As System.Diagnostics.StackFrame = New System.Diagnostics.StackFrame(0)
        For i As Integer = 0 To 5

            frame = New System.Diagnostics.StackFrame(i)
            If ((frame.GetMethod().Name <> "checkCanTpStatus") And (frame.GetMethod().Name <> "GetCallerMethodName")) Then
                Return frame.GetMethod().Name
            End If
        Next i
        Return "Unknown"

    End Function

    ''' <summary>
    ''' this Is the CCITT CRC 16 polynomial X^16  + X^12  + X^5  + 1.
    ''' This works out to be 0x1021, but the way the algorithm works
    ''' lets us use 0x8408 (the reverse of the bit pattern).  The high
    ''' bit Is always assumed to be set, thus we only use 16 bits to
    ''' represent the 17 bit value.
    ''' </summary>
    ''' <param name="p_data">Tab of BYTE</param>
    ''' <param name="p_length">Length of tab</param>
    ''' <returns>CRC 16</returns>
    Private Function crc16(ByVal p_data As Byte(), ByVal p_length As UShort) As UShort

        Dim i As Byte = 0
        Dim indexData As UShort = 0
        Dim data As UShort
        Dim crc As UShort = &HFFFF
        Const POLY As UShort = &H8408

        If (p_length = 0) Then
            Return System.Convert.ToByte(Not crc) ' Bitwise invert
        End If

        Do
            data = System.Convert.ToByte(&HFF And p_data(indexData))
            indexData += 1
            For i = 0 To 7

                If (((crc And &H1) Xor (data And &H1)) <> 0) Then
                    crc = System.Convert.ToUInt16((crc >> 1) Xor POLY)
                Else crc >>= 1
                End If
                data >>= 1
            Next i
            p_length -= 1
        Loop While (p_length > 0)

        crc = System.Convert.ToUInt16(Not crc)
        data = crc
        crc = System.Convert.ToUInt16((crc << 8) Or ((data >> 8) And &HFF))

        Return (crc)

    End Function
    ''' <summary>
    ''' Includes a new line of text into the information ListBox
    ''' </summary>
    ''' <param name="strMsg">Text to be included</param>
    Private Sub includeTextMessage(ByVal strMsg As String)
        listBoxParamInfo.Items.Add(strMsg)
        listBoxParamInfo.SelectedIndex = listBoxParamInfo.Items.Count - 1
    End Sub

    ''' <summary>
    ''' First initialization of UI components of the "Connection" tab.
    ''' </summary>
    Private Sub initializeUiConnection()
        ' fill combobox "comboBoxBaudrate"
        comboBoxBaudrate.Items.Clear()
        For Each value As TPCANTPBaudrate In [Enum].GetValues(GetType(TPCANTPBaudrate))
            comboBoxBaudrate.Items.Add(New ComboBoxItem(CanTpUtils.GetBitrate(value), value))
        Next
        comboBoxBaudrate.SelectedIndex = 2
        ' fill combobox "comboBoxHwType"
        comboBoxHwType.Items.Clear()
        For Each value As TPCANTPHWType In [Enum].GetValues(GetType(TPCANTPHWType))
            comboBoxHwType.Items.Add(New ComboBoxItem([Enum].GetName(GetType(TPCANTPHWType), value), value))
        Next
        ' fill combobox "comboBoxIoPort"
        comboBoxIoPort.Items.Clear()
        comboBoxIoPort.Items.Add(New ComboBoxItem("0100", &H100))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0120", &H120))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0140", &H140))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0200", &H200))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0220", &H220))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0240", &H240))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0260", &H260))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0278", &H278))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0280", &H280))
        comboBoxIoPort.Items.Add(New ComboBoxItem("02A0", &H2A0))
        comboBoxIoPort.Items.Add(New ComboBoxItem("02C0", &H2C0))
        comboBoxIoPort.Items.Add(New ComboBoxItem("02E0", &H2E0))
        comboBoxIoPort.Items.Add(New ComboBoxItem("02E8", &H2E8))
        comboBoxIoPort.Items.Add(New ComboBoxItem("02F8", &H2F8))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0300", &H300))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0320", &H320))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0340", &H340))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0360", &H360))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0378", &H378))
        comboBoxIoPort.Items.Add(New ComboBoxItem("0380", &H380))
        comboBoxIoPort.Items.Add(New ComboBoxItem("03BC", &H3BC))
        comboBoxIoPort.Items.Add(New ComboBoxItem("03E0", &H3E0))
        comboBoxIoPort.Items.Add(New ComboBoxItem("03E8", &H3E8))
        comboBoxIoPort.Items.Add(New ComboBoxItem("03F8", &H3F8))
        ' fill combobox "comboBoxInterrupt"
        comboBoxInterrupt.Items.Clear()
        comboBoxInterrupt.Items.Add(New ComboBoxItem("3", 3))
        comboBoxInterrupt.Items.Add(New ComboBoxItem("4", 4))
        comboBoxInterrupt.Items.Add(New ComboBoxItem("5", 5))
        comboBoxInterrupt.Items.Add(New ComboBoxItem("7", 7))
        comboBoxInterrupt.Items.Add(New ComboBoxItem("9", 9))
        comboBoxInterrupt.Items.Add(New ComboBoxItem("10", 10))
        comboBoxInterrupt.Items.Add(New ComboBoxItem("11", 11))
        comboBoxInterrupt.Items.Add(New ComboBoxItem("12", 12))
        comboBoxInterrupt.Items.Add(New ComboBoxItem("15", 15))
        ' fill combobox "comboBoxChannel"
        buttonHwRefresh_Click(Me, EventArgs.Empty)
        ' select last item
        comboBoxChannel.SelectedIndex = comboBoxChannel.Items.Count - 1
    End Sub
    ''' <summary>
    ''' First initialization of UI components of the "Message" tab.
    ''' </summary>
    Private Sub initializeUiTabMessages()
        numericUpDownMsgLength_ValueChanged(Me, EventArgs.Empty)
    End Sub
    ''' <summary>
    ''' First initialization of UI components of the "Parameter" tab.
    ''' </summary>
    Private Sub initializeUiTabParameters()
        Dim param As TPCANTPParameter
        ' fill combobox "comboBoxParameter"
        comboBoxParameter.Items.Clear()
        param = TPCANTPParameter.PCANTP_PARAM_API_VERSION
        comboBoxParameter.Items.Add(New ComboBoxItem(CanTpUtils.GetParameter(param), param))
        param = TPCANTPParameter.PCANTP_PARAM_CHANNEL_CONDITION
        comboBoxParameter.Items.Add(New ComboBoxItem(CanTpUtils.GetParameter(param), param))
        param = TPCANTPParameter.PCANTP_PARAM_DEBUG
        comboBoxParameter.Items.Add(New ComboBoxItem(CanTpUtils.GetParameter(param), param))

        param = TPCANTPParameter.PCANTP_PARAM_RECEIVE_EVENT
        comboBoxParameter.Items.Add(New ComboBoxItem(CanTpUtils.GetParameter(param), param))
        param = TPCANTPParameter.PCANTP_PARAM_MSG_PENDING
        comboBoxParameter.Items.Add(New ComboBoxItem(CanTpUtils.GetParameter(param), param))

        param = TPCANTPParameter.PCANTP_PARAM_BLOCK_SIZE
        comboBoxParameter.Items.Add(New ComboBoxItem(CanTpUtils.GetParameter(param), param))
        param = TPCANTPParameter.PCANTP_PARAM_SEPARATION_TIME
        comboBoxParameter.Items.Add(New ComboBoxItem(CanTpUtils.GetParameter(param), param))
        param = TPCANTPParameter.PCANTP_PARAM_WFT_MAX
        comboBoxParameter.Items.Add(New ComboBoxItem(CanTpUtils.GetParameter(param), param))

        param = TPCANTPParameter.PCANTP_PARAM_PADDING_VALUE
        comboBoxParameter.Items.Add(New ComboBoxItem(CanTpUtils.GetParameter(param), param))
        param = TPCANTPParameter.PCANTP_PARAM_CAN_DATA_PADDING
        comboBoxParameter.Items.Add(New ComboBoxItem(CanTpUtils.GetParameter(param), param))
        'J1939 Priority
        param = TPCANTPParameter.PCANTP_PARAM_J1939_PRIORITY
        comboBoxParameter.Items.Add(New ComboBoxItem(CanTpUtils.GetParameter(param), param))
        ' DLC when FD Is selected
        param = TPCANTPParameter.PCANTP_PARAM_CAN_TX_DL
        comboBoxParameter.Items.Add(New ComboBoxItem(CanTpUtils.GetParameter(param), param))
    End Sub

    ''' <summary>
    ''' Activates/deactivates the different controls of the main-form according
    ''' with the current connection status
    ''' </summary>
    ''' <param name="bConnected">Current status. True if connected, false otherwise</param>
    Private Sub setConnectionStatus(ByVal bConnected As Boolean)
        ' Connection components
        buttonInit.Enabled = Not bConnected
        buttonRelease.Enabled = bConnected
        buttonHwRefresh.Enabled = Not bConnected
        comboBoxChannel.Enabled = Not bConnected
        comboBoxBaudrate.Enabled = Not bConnected
        comboBoxHwType.Enabled = Not bConnected
        comboBoxIoPort.Enabled = Not bConnected
        comboBoxInterrupt.Enabled = Not bConnected
        ' Parameters components
        buttonParamReset.Enabled = bConnected
        buttonParamStatus.Enabled = bConnected
        comboBoxParameter_SelectedIndexChanged(Me, EventArgs.Empty)
        ' Mappings components
        buttonMappingAdd.Enabled = bConnected
        buttonMappingDel.Enabled = bConnected
        buttonMappingSample.Enabled = bConnected
        listViewMappings.Enabled = bConnected
        listViewMappings_SelectedIndexChanged(Me, EventArgs.Empty)
        ' Messages components
        buttonMsgRead.Enabled = bConnected
        buttonMsgWrite.Enabled = bConnected
        comboBoxMsgMapping.Enabled = bConnected
        listViewMsgs.Enabled = bConnected
        comboBoxMsgMapping_SelectedIndexChanged(Me, EventArgs.Empty)
        If Not bConnected Then
            ' reload hardware configuration if NOT connected
            comboBoxChannel_SelectedIndexChanged(Me, EventArgs.Empty)
        Else
            ' update reading method if connected
            radioButtonMsgRead_CheckedChanged(Me, EventArgs.Empty)
        End If
    End Sub

    ''' <summary>
    ''' States if an ISO-TP channel is currently connected.
    ''' </summary>
    ''' <returns>true if a channel is connected.</returns>
    Private Function isConnected() As Boolean
        Return m_pctpHandle <> CanTpApi.PCANTP_NONEBUS
    End Function
#End Region

    ''' <summary>
    ''' Represents a PCAN device
    ''' </summary>
    Public Enum TPCANDevice As Byte
        ''' <summary>
        ''' Undefined, unknown or not selected PCAN device value
        ''' </summary>
        PCAN_NONE = 0
        ''' <summary>
        ''' PCAN Non-Plug and Play devices. NOT USED WITHIN PCAN-Basic API
        ''' </summary>
        PCAN_PEAKCAN = 1
        ''' <summary>
        ''' PCAN-ISA, PCAN-PC/104, and PCAN-PC/104-Plus
        ''' </summary>
        PCAN_ISA = 2
        ''' <summary>
        ''' PCAN-Dongle
        ''' </summary>
        PCAN_DNG = 3
        ''' <summary>
        ''' PCAN-PCI, PCAN-cPCI, PCAN-miniPCI, and PCAN-PCI Express
        ''' </summary>
        PCAN_PCI = 4
        ''' <summary>
        ''' PCAN-USB and PCAN-USB Pro
        ''' </summary>
        PCAN_USB = 5
        ''' <summary>
        ''' PCAN-PC Card
        ''' </summary>
        PCAN_PCC = 6
        ''' <summary>
        ''' PCAN Virtual hardware. NOT USED WITHIN PCAN-Basic API
        ''' </summary>
        PCAN_VIRTUAL = 7
        ''' <summary>
        ''' PCAN Gateway devices
        ''' </summary>
        PCAN_LAN = 8
    End Enum

    ''' <summary>
    ''' Represents an item of a ComboBox
    ''' </summary>
    Class ComboBoxItem
        ''' <summary>
        ''' Real data object to add to the comboBox
        ''' </summary>
        Public Property Data() As Object
        ''' <summary>
        ''' Text to display in the comboBox
        ''' </summary>
        Public Property Text() As String
        ''' <summary>
        ''' Default constructor
        ''' </summary>
        ''' <param name="text">Text to display.</param>
        ''' <param name="data">Real data object.</param>
        Public Sub New(ByVal text As String, ByVal data As Object)
            Me.Text = text
            Me.Data = data
        End Sub
        Public Overrides Function ToString() As String
            Return Text
        End Function
    End Class

    ''' <summary>
    ''' Class that stores information of a single ISO-TP mapping.
    ''' </summary>
    Public Class MappingStatus
        ''' <summary>
        ''' CAN ID to map to CAN-ISO-TP network addressing information
        ''' </summary>
        Public Property m_canId() As UInteger
        ''' <summary>
        ''' CAN ID used by the other side to internally respond to the CAN-ISO-TP segmented frames
        ''' (i.e. the Flow Control frames will use this ID)
        ''' </summary>
        Public Property m_canIdResponse() As UInteger
        ''' <summary>
        ''' The CAN ID type used by the mapping (11 bits or 29 bits CAN ID)
        ''' </summary>
        Public Property m_canIdType() As TPCANTPIdType
        ''' <summary>
        ''' The ISO-TP network addressing format.
        ''' </summary>
        Public Property m_formatType() As TPCANTPFormatType
        ''' <summary>
        ''' Type of CAN-ISO-TP message (diagnostic or remote disgnostic message)
        ''' </summary>
        Public Property m_msgType() As TPCANTPMessageType
        ''' <summary>
        ''' Source address
        ''' </summary>
        Public Property m_sourceAddr() As Byte
        ''' <summary>
        ''' Target address
        ''' </summary>
        Public Property m_targetAddr() As Byte
        ''' <summary>
        ''' Type of addressing (physical: -node to node-, or functional: -node to all-)
        ''' </summary>
        Public Property m_targetType() As TPCANTPAddressingType
        ''' <summary>
        ''' Remote address (used only with remote disgnostic message)
        ''' </summary>
        Public Property m_remoteAddr() As Byte

        ''' <summary>
        ''' Default constructor
        ''' </summary>
        ''' <param name="canId">CAN ID to map to CAN-ISO-TP network addressing information</param>
        ''' <param name="canIdResponse">CAN ID used by the other side to internally respond to the CAN-ISO-TP segmented frames
        ''' (i.e. the Flow Control frames will use this ID)</param>
        ''' <param name="canIdType">The CAN ID type used by the mapping (11 bits or 29 bits CAN ID)</param>
        ''' <param name="formatType">The ISO-TP network addressing format.</param>
        ''' <param name="msgType">Type of CAN-ISO-TP message (diagnostic or remote disgnostic message)</param>
        ''' <param name="sourceAddr">Source address</param>
        ''' <param name="targetAddr">Target address</param>
        ''' <param name="targetType">Type of addressing (physical: -node to node-, or functional: -node to all-)</param>
        ''' <param name="remoteAddr">Remote address (used only with remote disgnostic message)</param>
        Public Sub New(
            Optional ByVal canId As UInteger = &H0,
            Optional ByVal canIdResponse As UInteger = &H0,
            Optional ByVal canIdType As TPCANTPIdType = TPCANTPIdType.PCANTP_ID_CAN_11BIT,
            Optional ByVal formatType As TPCANTPFormatType = TPCANTPFormatType.PCANTP_FORMAT_UNKNOWN,
            Optional ByVal msgType As TPCANTPMessageType = TPCANTPMessageType.PCANTP_MESSAGE_UNKNOWN,
            Optional ByVal targetType As TPCANTPAddressingType = TPCANTPAddressingType.PCANTP_ADDRESSING_UNKNOWN,
            Optional ByVal sourceAddr As Byte = &H0,
            Optional ByVal targetAddr As Byte = &H0,
            Optional ByVal remoteAddr As Byte = &H0)
            m_canId = canId
            m_canIdResponse = canIdResponse
            m_canIdType = canIdType
            m_formatType = formatType
            m_msgType = msgType
            m_sourceAddr = sourceAddr
            m_targetAddr = targetAddr
            m_targetType = targetType
            m_remoteAddr = remoteAddr
        End Sub

        ''' <summary>
        ''' Gets a new instance of the object.
        ''' </summary>
        ''' <returns>The cloned instance of the object.</returns>
        Public Function Clone() As MappingStatus
            Return New MappingStatus(m_canId, m_canIdResponse, m_canIdType,
                m_formatType, m_msgType, m_targetType, m_sourceAddr, m_targetAddr, m_remoteAddr)
        End Function
        ''' <summary>
        ''' States if an ISO-TP mapping is NOT required for this configuration.
        ''' Some 29 bits CAN ID ISO-TP message with specific format addressing 
        ''' do NOT require ISO-TP mappings.
        ''' </summary>
        Public ReadOnly Property IsAutomatic() As Boolean
            Get
                If (m_canIdType = TPCANTPIdType.PCANTP_ID_CAN_29BIT AndAlso (
                    Me.m_formatType = TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL OrElse
                    Me.m_formatType = TPCANTPFormatType.PCANTP_FORMAT_MIXED OrElse
                    Me.m_formatType = TPCANTPFormatType.PCANTP_FORMAT_ENHANCED)
                    ) Then
                    Return True
                End If
                Return False
            End Get
        End Property

        ''' <summary>
        ''' The formatted name of this mapping configuration.
        ''' </summary>
        Public ReadOnly Property Name As String
            Get
                Dim name_ As String
                name_ = CanIdType + ", " + FormatType + ", " + TargetType

                If (Not IsAutomatic) Then
                    name_ += ": " + SourceAddress + " -> " + TargetAddress
                End If
                Return name_
            End Get
        End Property
        ''' <summary>
        ''' The CAN ID of the configured mapping as a string.
        ''' Note: automatic mapping has no CAN ID, since it is 
        ''' handled on the fly by the ISO-TP API.
        ''' </summary>
        Public ReadOnly Property CanId As String
            Get
                If (IsAutomatic) Then
                    Return "-"
                End If
                Return CanTpUtils.GetCanId(m_canId, m_canIdType = TPCANTPIdType.PCANTP_ID_CAN_29BIT)
            End Get
        End Property
        ''' <summary>
        ''' The CAN ID response of the configured mapping as a string.
        ''' Note: automatic mapping has no CAN ID response, since it is 
        ''' handled on the fly by the ISO-TP API.
        ''' </summary>
        Public ReadOnly Property CanIdResponse As String
            Get
                If (IsAutomatic) Then
                    Return "-"
                End If
                Return CanTpUtils.GetCanId(m_canIdResponse, m_canIdType = TPCANTPIdType.PCANTP_ID_CAN_29BIT)
            End Get
        End Property
        ''' <summary>
        ''' The CAN ID type of the configured mapping as a string.
        ''' </summary>
        Public ReadOnly Property CanIdType As String
            Get
                Return CanTpUtils.GetCanIdType(m_canIdType)
            End Get
        End Property
        ''' <summary>
        ''' The ISO-TP message type of the configured mapping as a string.
        ''' </summary>
        Public ReadOnly Property MsgType As String
            Get
                Return CanTpUtils.GetMsgType(m_msgType)
            End Get
        End Property
        ''' <summary>
        ''' The ISO-TP addressing format type of the configured mapping as a string.
        ''' </summary>
        Public ReadOnly Property FormatType As String
            Get
                Return CanTpUtils.GetFormatType(m_formatType)
            End Get
        End Property
        ''' <summary>
        ''' The ISO-TP target address type ID type of the configured mapping as a string.
        ''' </summary>
        Public ReadOnly Property TargetType As String
            Get
                Return CanTpUtils.GetTargetType(m_targetType)
            End Get
        End Property
        ''' <summary>
        ''' The source address of the configured mapping as a string.
        ''' </summary>
        Public ReadOnly Property SourceAddress As String
            Get
                If (IsAutomatic) Then
                    Return "-"
                End If
                Return CanTpUtils.GetAddress(m_sourceAddr)
            End Get
        End Property
        ''' <summary>
        ''' The target address of the configured mapping as a string.
        ''' </summary>
        Public ReadOnly Property TargetAddress As String
            Get
                If (IsAutomatic) Then
                    Return "-"
                End If
                Return CanTpUtils.GetAddress(m_targetAddr)
            End Get
        End Property
        ''' <summary>
        ''' The remote address of the configured mapping as a string.
        ''' </summary>
        Public ReadOnly Property RemoteAddress As String
            Get
                If (IsAutomatic) Then
                    Return "-"
                End If
                Return CanTpUtils.GetAddress(m_remoteAddr)
            End Get
        End Property
    End Class

    ''' <summary>
    ''' Class that stores information on a received ISO-TP message.
    ''' </summary>
    Public Class MessageStatus
        ''' <summary>
        ''' The last ISO-TP message received.
        ''' </summary>
        Private m_msg As TPCANTPMsg
        ''' <summary>
        ''' The timestamp of the last ISO-TP message received.
        ''' </summary>
        Private m_timeStamp As TPCANTPTimestamp
        ''' <summary>
        ''' The timestamp of the before last ISO-TP message received.
        ''' </summary>
        Private m_timeStampOld As TPCANTPTimestamp
        ''' <summary>
        ''' Index of the object in the message listView.
        ''' </summary>
        Private m_index As Integer
        ''' <summary>
        ''' Number of similar ISO-TP message received 
        ''' (i.e. messages having the same Network Address Information).
        ''' </summary>
        Private m_count As Integer
        ''' <summary>
        ''' States if the timestamp should be displayed as a period.
        ''' </summary>
        Private m_showPeriod As Boolean
        ''' <summary>
        ''' States if the object has been modified but not updated/displayed on the UI.
        ''' </summary>
        Private m_wasChanged As Boolean

        ''' <summary>
        ''' Default constructor.
        ''' </summary>
        ''' <param name="canTpMsg">The received ISO-TP message.</param>
        ''' <param name="canTpTimestamp">The timestamp when the message was received.</param>
        ''' <param name="listIndex">Position of the messageStatus in the message listView.</param>
        Public Sub New(ByVal canTpMsg As TPCANTPMsg, ByVal canTpTimestamp As TPCANTPTimestamp, ByVal listIndex As Integer)
            m_msg = canTpMsg
            m_timeStamp = canTpTimestamp
            m_timeStampOld = canTpTimestamp
            m_index = listIndex
            m_count = 1
            m_showPeriod = True
            m_wasChanged = False
        End Sub

        ''' <summary>
        ''' Updates the messageStatus' information with a newly received ISO-TP message.
        ''' </summary>
        ''' <param name="canTpMsg">The received ISO-TP message.</param>
        ''' <param name="canTpTimestamp">The timestamp when the message was received.</param>
        Public Sub update(ByVal canTpMsg As TPCANTPMsg, ByVal canTpTimestamp As TPCANTPTimestamp)
            m_msg = canTpMsg
            m_timeStampOld = m_timeStamp
            m_timeStamp = canTpTimestamp
            m_wasChanged = True
            m_count += 1
        End Sub

        ''' <summary>
        ''' States if a message has the same network address information as 
        ''' the last ISO-TP message received.
        ''' </summary>
        ''' <param name="canTpMsg">The ISO-TP message to compare to.</param>
        ''' <returns></returns>
        Public Function isSameNetAddrInfo(ByVal canTpMsg As TPCANTPMsg) As Boolean
            Return (m_msg.SA = canTpMsg.SA AndAlso
                m_msg.TA = canTpMsg.TA AndAlso
                m_msg.RA = canTpMsg.RA AndAlso
                m_msg.IDTYPE = canTpMsg.IDTYPE AndAlso
                m_msg.FORMAT = canTpMsg.FORMAT AndAlso
                m_msg.TA_TYPE = canTpMsg.TA_TYPE AndAlso
                m_msg.MSGTYPE = canTpMsg.MSGTYPE)
        End Function

#Region "Getters / Setters"
        ''' <summary>
        ''' The last ISO-TP message received.
        ''' </summary>
        Public ReadOnly Property CanTpMsg As TPCANTPMsg
            Get
                Return m_msg
            End Get
        End Property
        ''' <summary>
        ''' The timestamp of the last ISO-TP message received.
        ''' </summary>
        Public ReadOnly Property Timestamp As TPCANTPTimestamp
            Get
                Return m_timeStamp
            End Get
        End Property
        ''' <summary>
        ''' The index of the object in the "received message" listView.
        ''' </summary>
        Public ReadOnly Property Position As Integer
            Get
                Return m_index
            End Get
        End Property
        ''' <summary>
        ''' Number of time a similar message was received (same network address information).
        ''' </summary>
        Public ReadOnly Property Count As Integer
            Get
                Return m_count
            End Get
        End Property
        ''' <summary>
        ''' States if the timestamp should be displayed as a period.
        ''' </summary>
        Public Property ShowingPeriod As Boolean
            Get
                Return m_showPeriod
            End Get
            Set(ByVal value As Boolean)
                If m_showPeriod Xor value Then
                    m_showPeriod = value
                    m_wasChanged = True
                End If
            End Set
        End Property
        ''' <summary>
        ''' States if the object was modified since last display.
        ''' </summary>
        Public Property MarkedAsUpdated As Boolean
            Get
                Return m_wasChanged
            End Get
            Set(ByVal value As Boolean)
                m_wasChanged = value
            End Set
        End Property
        ''' <summary>
        ''' Unique ID as a string of the last message received.
        ''' </summary>
        Public ReadOnly Property UID As String
            Get
                Return CanTpUtils.GetUniqueId(m_msg)
            End Get
        End Property
        ''' <summary>
        ''' CAN ID type as a string of the last message received.
        ''' </summary>
        Public ReadOnly Property CanIdType As String
            Get
                Return CanTpUtils.GetCanIdType(m_msg.IDTYPE)
            End Get
        End Property
        ''' <summary>
        ''' ISO-TP Message type as a string of the last message received.
        ''' </summary>
        Public ReadOnly Property MsgType As String
            Get
                Return CanTpUtils.GetMsgType(m_msg.MSGTYPE)
            End Get
        End Property
        ''' <summary>
        ''' ISO-TP addressing format type as a string of the last message received.
        ''' </summary>
        Public ReadOnly Property FormatType As String
            Get
                Return CanTpUtils.GetFormatType(m_msg.FORMAT)
            End Get
        End Property
        ''' <summary>
        ''' ISO-TP target addressing type as a string of the last message received.
        ''' </summary>
        Public ReadOnly Property TargetType As String
            Get
                Return CanTpUtils.GetTargetType(m_msg.TA_TYPE)
            End Get
        End Property
        ''' <summary>
        ''' ISO-TP source address as a string of the last message received.
        ''' </summary>
        Public ReadOnly Property SourceAddress As String
            Get
                If m_msg.FORMAT = TPCANTPFormatType.PCANTP_FORMAT_ENHANCED Then
                    ' Unique SA is 11bits: store bits [7..0] in SA
                    ' and the rest in [3..0] of RA
                    Return CanTpUtils.GetCanId(CUInt((m_msg.RA >> 4) And &H7) <<8 Or m_msg.SA, False)
                End If
        Return CanTpUtils.GetAddress(m_msg.SA)
            End Get
        End Property
        ''' <summary>
        ''' ISO-TP target address as a string of the last message received.
        ''' </summary>
        Public ReadOnly Property TargetAddress As String
            Get
                If m_msg.FORMAT = TPCANTPFormatType.PCANTP_FORMAT_ENHANCED Then
                    ' Unique TA is 11bits: store bits [7..0] in TA
                    ' and the rest in [7..4] of RA
                    Return CanTpUtils.GetCanId(CUInt(m_msg.RA And &H7) << 8 Or m_msg.TA, False)
                End If
                Return CanTpUtils.GetAddress(m_msg.TA)
            End Get
        End Property
        ''' <summary>
        ''' ISO-TP remote address as a string of the last message received.
        ''' </summary>
        Public ReadOnly Property RemoteAddress As String
            Get
                Return CanTpUtils.GetAddress(m_msg.RA)
            End Get
        End Property
        ''' <summary>
        ''' Lenght of the data as a string of the last message received.
        ''' </summary>
        Public ReadOnly Property LengthString As String
            Get
                Return m_msg.LEN.ToString()
            End Get
        End Property
        ''' <summary>
        ''' Data as a string of the last message received.
        ''' </summary>
        Public ReadOnly Property DataString As String
            Get
                Return CanTpUtils.GetDataString(m_msg)
            End Get
        End Property
        ''' <summary>
        ''' ISO-TP Network result as a string of the last message received.
        ''' </summary>
        Public ReadOnly Property ResultString As String
            Get
                Return CanTpUtils.GetResult(m_msg.RESULT)
            End Get
        End Property
        ''' <summary>
        ''' Timestamp (or period) as a string of the last message received.
        ''' </summary>
        Public ReadOnly Property TimeString As String
            Get
                Return CanTpUtils.GetTimeString(m_timeStamp, m_showPeriod, m_timeStampOld)
            End Get
        End Property
#End Region

    End Class

    Private Sub buttonMappingLoad_Click(sender As Object, e As EventArgs) Handles buttonMappingLoad.Click
        readMappingFile()
    End Sub
    Public Function readMappingFile() As Boolean
        Dim mapping As MappingStatus
        Try
            If (System.IO.File.Exists("Mappings.csv")) Then

                ' Open the file to read from.
                Dim readText() As String = System.IO.File.ReadAllLines("Mappings.csv")
                Dim skipFirstLine As Boolean = True ' First line Is CSV header
                Dim s As String
                For Each s In readText

                    If (skipFirstLine = True) Then
                        skipFirstLine = False
                        Continue For
                    End If
                    Dim allValues As String() = s.Split(";")
                    If (allValues.Length = 9) Then ' mapping Then must have 9 elements To be valid

                        '            0      1                  2           3    4  5   6       7     8
                        'newLine = "CAN_ID;CAN_ID response;Target Type;ID Type;SA;TA;MsgType;Format;RA" + Environment.NewLine;
                        'mapping = New MappingStatus(0, 1, 3, 7, 6, 2, 4, 5, 8)
                        mapping = New MappingStatus(allValues(0),
                                                    allValues(1),
                                                    allValues(3),
                                                    allValues(7),
                                                    allValues(6),
                                                    allValues(2),
                                                    allValues(4),
                                                    allValues(5),
                                                    allValues(8))
                        mappingsAdd(mapping)
                        mappingsChanged()
                    End If
                Next
            End If
        Catch ex As Exception

        End Try

        Return True

    End Function

    Private Sub buttonMappingSave_Click(sender As Object, e As EventArgs) Handles buttonMappingSave.Click
        Try

            Dim CSVHeader As String = "CAN_ID;CAN_ID response;Target Type;ID Type;SA;TA;MsgType;Format;RA" + Environment.NewLine
            Dim newLine As String = ""
            Using mappingFileWriter As System.IO.StreamWriter = New System.IO.StreamWriter("Mappings.csv", False)
                mappingFileWriter.WriteLine(CSVHeader)
                For i As Integer = 5 To m_mappings.Count - 1 ' Skip 5 firsts defaults mappings
                    newLine = Convert.ToString(m_mappings(i).m_canId) + ";" + Convert.ToString(m_mappings(i).m_canIdResponse) + ";" + Convert.ToString(m_mappings(i).m_targetType) +
                           ";" + Convert.ToString(m_mappings(i).m_canIdType) + ";" + Convert.ToString(m_mappings(i).m_sourceAddr) +
                            ";" + Convert.ToString(m_mappings(i).m_targetAddr) + ";" + Convert.ToString(m_mappings(i).m_msgType) +
                          ";" + Convert.ToString(m_mappings(i).m_formatType) + ";" + Convert.ToString(m_mappings(i).m_remoteAddr)
                    mappingFileWriter.WriteLine(newLine)
                Next
            End Using
        Catch ex As Exception

        End Try
    End Sub

    Private Sub checkBoxCanFd_CheckedChanged(sender As Object, e As EventArgs) Handles checkBoxCanFd.CheckedChanged
        Dim bIsChecked As Boolean = checkBoxCanFd.Checked

        labelConnBaudrate.Visible = Not bIsChecked
        labelConnIoPort.Visible = Not bIsChecked
        labelConnHwType.Visible = Not bIsChecked
        labelConnInterrupt.Visible = Not bIsChecked

        comboBoxBaudrate.Visible = Not bIsChecked
        comboBoxIoPort.Visible = Not bIsChecked
        comboBoxHwType.Visible = Not bIsChecked
        comboBoxInterrupt.Visible = Not bIsChecked

        textBoxCanFdBitrate.Visible = bIsChecked
        labelConnBitRate.Visible = bIsChecked

        ' Manage FD, BRS And Priority checkboxes (in messages tabulation)
        checkBoxFDMessage.Visible = bIsChecked
        checkBoxBRS.Visible = bIsChecked
        checkBoxFDMessage.Checked = False
        checkBoxHasPriority.Checked = False
        checkBoxBRS.Checked = False
        ManageUIForFD(bIsChecked)

    End Sub


    ''' <summary>
    ''' Manage FD checkboxes when FD support Is checked
    ''' </summary>
    Private Sub ManageUIForFD(p_state As Boolean)
        checkBoxBRS.Visible = p_state
        checkBoxFDMessage.Visible = p_state
        checkBoxBRS.Enabled = Not p_state
        checkBoxFDMessage.Enabled = p_state
    End Sub

    Private Sub checkBoxFDMessage_CheckedChanged(sender As Object, e As EventArgs) Handles checkBoxFDMessage.CheckedChanged
        checkBoxBRS.Enabled = If(checkBoxFDMessage.Checked = True, True, False)
    End Sub

    Private Sub checkBoxHasPriority_CheckedChanged(sender As Object, e As EventArgs) Handles checkBoxHasPriority.CheckedChanged
        numericUpDownPriority.Enabled = checkBoxHasPriority.Checked
    End Sub
End Class


''' <summary>
''' A module providing various fonctions to format ISO-TP data.
''' </summary>
Public Module CanTpUtils
    ''' <summary>
    ''' Maximum data length of a physical ISO-TP message.
    ''' </summary>
    Public Const ISOTP_MSG_PHYS_MAX_LENGTH As Integer = 4095
    ''' <summary>
    ''' Maximum data length of a functional ISO-TP message.
    ''' </summary>
    Public Const ISOTP_MSG_FUNC_MAX_LENGTH As Integer = 7

    ''' <summary>
    ''' Gets the formated text from a PCAN-ISO-TP Btr0/Btr1 code.
    ''' </summary>
    ''' <param name="value">PCAN-ISO-TP Baudrate to format</param>
    ''' <returns>The formatted text for a baudrate</returns>
    Public Function GetBitrate(ByVal value As TPCANTPBaudrate) As String
        Select Case value
            Case TPCANTPBaudrate.PCANTP_BAUD_1M
                Return "1 MBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_800K
                Return "800 kBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_500K
                Return "500 kBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_250K
                Return "250 kBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_125K
                Return "125 kBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_100K
                Return "100 kBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_95K
                Return "95,238 kBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_83K
                Return "83,333 kBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_50K
                Return "50 kBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_47K
                Return "47,619 kBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_33K
                Return "33,333 kBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_20K
                Return "20 kBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_10K
                Return "10 kBit/s"
            Case TPCANTPBaudrate.PCANTP_BAUD_5K
                Return "5 kBit/s"
        End Select
        Return value.ToString()
    End Function
    ''' <summary>
    ''' Gets the formated text for a PCAN-ISO-TP channel handle.
    ''' </summary>
    ''' <param name="handle">PCAN-ISO-TP Handle to format</param>
    ''' <returns>The formatted text for a channel</returns>
    Public Function GetChannelName(ByVal handle As TPCANTPHandle) As String
        Dim devDevice As FormMain.TPCANDevice
        Dim byChannel As Byte

        ' Gets the owner device and channel for a 
        ' PCAN-Basic handle
        If handle < &H100 Then
            devDevice = DirectCast(CByte(handle >> 4), FormMain.TPCANDevice)
            byChannel = CByte(handle And &HF)
        Else
            devDevice = DirectCast(CByte(handle >> 8), FormMain.TPCANDevice)
            byChannel = CByte(handle And &HFF)
        End If
        ' Constructs the PCAN-Basic Channel name and return it
        Return String.Format("{0} {1} ({2:X2}h)", devDevice, byChannel, handle)
    End Function
    ''' <summary>
    ''' Gets the formated text from a TPCANTPParameter value.
    ''' </summary>
    ''' <param name="param">Parameter to format.</param>
    ''' <returns>The parameter as a text.</returns>
    Public Function GetParameter(ByVal param As TPCANTPParameter) As String
        Select Case param
            Case TPCANTPParameter.PCANTP_PARAM_API_VERSION
                Return "API version"
            Case TPCANTPParameter.PCANTP_PARAM_BLOCK_SIZE
                Return "ISO-TP Block Size (BS)"
            Case TPCANTPParameter.PCANTP_PARAM_CAN_DATA_PADDING
                Return "CAN Data Padding"
            Case TPCANTPParameter.PCANTP_PARAM_CHANNEL_CONDITION
                Return "Channel condition"
            Case TPCANTPParameter.PCANTP_PARAM_DEBUG
                Return "Debug mode"
            Case TPCANTPParameter.PCANTP_PARAM_MSG_PENDING
                Return "Message pending notification"
            Case TPCANTPParameter.PCANTP_PARAM_PADDING_VALUE
                Return "CAN Data Padding value"
            Case TPCANTPParameter.PCANTP_PARAM_RECEIVE_EVENT
                Return "Receive event"
            Case TPCANTPParameter.PCANTP_PARAM_SEPARATION_TIME
                Return "ISO-TP Separation time (STMin)"
            Case TPCANTPParameter.PCANTP_PARAM_WFT_MAX
                Return "ISO-TP FC.Wait frame max. (N_WFTmax)"
            Case TPCANTPParameter.PCANTP_PARAM_J1939_PRIORITY
                Return "ISO-TP J1939 priority"
            Case TPCANTPParameter.PCANTP_PARAM_CAN_TX_DL
                Return "ISO-TP FD Data Length Code (DLC)"
        End Select
        Return "Unknown parameter"
    End Function
    ''' <summary>
    ''' Gets the formatted text of a TPCANTPStatus error.
    ''' </summary>
    ''' <param name="error_">Error code to be translated</param>
    ''' <returns>A text with the translated error</returns>
    Public Function GetError(ByVal error_ As TPCANTPStatus) As String
        Dim strTemp As StringBuilder

        ' Creates a buffer big enough for a error-text
        strTemp = New StringBuilder(256)
        ' Gets the text using the GetErrorText API function
        ' If the function is successful, the translated error is returned. 
        ' If it fails, a text describing the current error is returned.
        If CanTpApi.GetErrorText(error_, 0, strTemp) <> TPCANTPStatus.PCANTP_ERROR_OK Then
            Return String.Format("An error occurred. Error-code's text ({0:X}) couldn't be retrieved", error_)
        Else
            Return strTemp.ToString()
        End If
    End Function
    ''' <summary>
    ''' Gets the formated text of a TPCANTPFormatType value.
    ''' </summary>
    ''' <param name="format">value to format</param>
    ''' <returns>The parameter formatted as a human-readable string.</returns>
    Public Function GetFormatType(ByVal format As TPCANTPFormatType) As String
        Select Case format
            Case TPCANTPFormatType.PCANTP_FORMAT_ENHANCED
                Return "Enhanced"
            Case TPCANTPFormatType.PCANTP_FORMAT_EXTENDED
                Return "Extended"
            Case TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL
                Return "Fixed Normal"
            Case TPCANTPFormatType.PCANTP_FORMAT_MIXED
                Return "Mixed"
            Case TPCANTPFormatType.PCANTP_FORMAT_NORMAL
                Return "Normal"
            Case Else
                Return "Unknown"
        End Select
    End Function
    ''' <summary>
    ''' Gets the formated text of a timestamp.
    ''' </summary>
    ''' <param name="ts">Timestamp to format</param>
    ''' <param name="showPeriod">States if the output is the period between 2 timestamps.</param>
    ''' <param name="tsOld">If showPeriod is true, the period is based on that previous timestamp</param>
    ''' <returns>The parameter formatted as a human-readable string.</returns>
    Public Function GetTimeString(ByVal ts As TPCANTPTimestamp, ByVal showPeriod As Boolean, ByVal tsOld As TPCANTPTimestamp) As String
        Dim fTime As Double

        fTime = ts.millis + (ts.micros / 1000.0)
        If showPeriod Then
            fTime -= (tsOld.millis + (tsOld.micros / 1000.0))
        End If
        Return fTime.ToString("F1")
    End Function
    ''' <summary>
    ''' Gets the data of an ISO-TP message as a formatted string.
    ''' </summary>
    ''' <param name="msg">ISO-TP message holding the data to format.</param>
    ''' <returns>The parameter formatted as a human-readable string.</returns>
    Public Function GetDataString(ByVal msg As TPCANTPMsg) As String
        Dim strTemp As String

        strTemp = ""
        For i As Integer = 0 To msg.LEN - 1
            strTemp += String.Format("{0:X2} ", msg.DATA(i))
        Next
        Return strTemp
    End Function
    ''' <summary>
    ''' Gets a Unique Identifier based of an ISO-TP message as a formatted string.
    ''' </summary>
    ''' <param name="msg">The ISO-TP message.</param>
    ''' <returns>The parameter formatted as a human-readable string.</returns>
    Public Function GetUniqueId(ByVal msg As TPCANTPMsg) As String
        ' We format the ID of the message and show it
        Return String.Format("{0:X2}h_{1:X2}h_{2}_{3}{4}_{5}_{6:X2}h",
            msg.SA, msg.TA,
            GetCanIdType(msg.IDTYPE),
            GetTargetType(msg.TA_TYPE),
            GetMsgType(msg.MSGTYPE), GetFormatType(msg.FORMAT), msg.RA)
    End Function
    ''' <summary>
    ''' Gets the CAN ID type as a formatted string.
    ''' </summary>
    ''' <param name="type">The can id type to format.</param>
    ''' <returns>The parameter formatted as a human-readable string.</returns>
    Public Function GetCanIdType(ByVal type As TPCANTPIdType) As String
        Dim strIdTypeReturn As String = ""
        If ((type And TPCANTPIdType.PCANTP_ID_CAN_11BIT) = TPCANTPIdType.PCANTP_ID_CAN_11BIT) Then
            strIdTypeReturn = "11bits "
        End If
        If ((type And TPCANTPIdType.PCANTP_ID_CAN_29BIT) = TPCANTPIdType.PCANTP_ID_CAN_29BIT) Then
            strIdTypeReturn += "29bits "
        End If
        If ((type And TPCANTPIdType.PCANTP_ID_CAN_FD) = TPCANTPIdType.PCANTP_ID_CAN_FD) Then
            strIdTypeReturn += "FD "
        End If
        If ((type And TPCANTPIdType.PCANTP_ID_CAN_BRS) = TPCANTPIdType.PCANTP_ID_CAN_BRS) Then
            strIdTypeReturn += "BRS "
        End If
        If (strIdTypeReturn = "") Then
            strIdTypeReturn = "Unknown"
        End If
        Return strIdTypeReturn
    End Function
    ''' <summary>
    ''' Gets the ISO-TP Addressing type as a formatted string.
    ''' </summary>
    ''' <param name="type">The addressing type to format.</param>
    ''' <returns>The parameter formatted as a human-readable string.</returns>
    Public Function GetTargetType(ByVal type As TPCANTPAddressingType) As String
        Select Case type
            Case TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL
                Return "Functional"
            Case TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL
                Return "Physical"
            Case Else
                Return "Unknown"
        End Select
    End Function
    ''' <summary>
    ''' Gets the ISO-TP message type as a formatted string.
    ''' </summary>
    ''' <param name="type">The message type to format.</param>
    ''' <returns>The parameter formatted as a human-readable string.</returns>
    Public Function GetMsgType(ByVal type As TPCANTPMessageType) As String
        Select Case type
            Case TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC
                Return "Diagnostic"
            Case TPCANTPMessageType.PCANTP_MESSAGE_INDICATION
                Return "Indication"
            Case TPCANTPMessageType.PCANTP_MESSAGE_INDICATION_TX
                Return "Indication TX"
            Case TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC
                Return "Remote Diagnostic"
            Case TPCANTPMessageType.PCANTP_MESSAGE_REQUEST_CONFIRMATION
                Return "Confirmation"
            Case Else
                Return "Unknown"
        End Select
    End Function
    ''' <summary>
    ''' Gets an ISO-TP address type as a formatted string.
    ''' </summary>
    ''' <param name="address">The address to format.</param>
    ''' <returns>The parameter formatted as a human-readable string.</returns>
    Public Function GetAddress(ByVal address As Byte) As String
        Return String.Format("{0:X2}h", address)
    End Function
    ''' <summary>
    ''' Gets an CAN ID as a formatted string.
    ''' </summary>
    ''' <param name="canId">The CAN ID to format.</param>
    ''' <param name="isExtended">True if the CAN ID is 29bits.</param>
    ''' <returns>The parameter formatted as a human-readable string.</returns>
    Public Function GetCanId(ByVal canId As UInteger, ByVal isExtended As Boolean) As String
        If Not isExtended Then
            Return String.Format("{0:X3}h", canId)
        Else
            Return String.Format("{0:X8}h", canId)
        End If
    End Function
    ''' <summary>
    ''' Gets an ISO-TP Network Result as a formatted string.
    ''' </summary>
    ''' <param name="result">The result to format.</param>
    ''' <returns>The parameter formatted as a human-readable string.</returns>
    Public Function GetResult(ByVal result As TPCANTPConfirmation) As String
        Dim strTmp As String
        Select Case result
            Case TPCANTPConfirmation.PCANTP_N_BUFFER_OVFLW
                strTmp = "Buffer overflow"
                Exit Select
            Case TPCANTPConfirmation.PCANTP_N_ERROR
                strTmp = "Error"
                Exit Select
            Case TPCANTPConfirmation.PCANTP_N_INVALID_FS
                strTmp = "Invalid First frame"
                Exit Select
            Case TPCANTPConfirmation.PCANTP_N_OK
                strTmp = "Ok"
                Exit Select
            Case TPCANTPConfirmation.PCANTP_N_TIMEOUT_A
                strTmp = "Timeout A"
                Exit Select
            Case TPCANTPConfirmation.PCANTP_N_TIMEOUT_BS
                strTmp = "Timeout Bs"
                Exit Select
            Case TPCANTPConfirmation.PCANTP_N_TIMEOUT_CR
                strTmp = "Timeout Cr"
                Exit Select
            Case TPCANTPConfirmation.PCANTP_N_UNEXP_PDU
                strTmp = "Unexpected Protocol Data Unit"
                Exit Select
            Case TPCANTPConfirmation.PCANTP_N_WFT_OVRN
                strTmp = "Wait For Transmit overrun"
                Exit Select
            Case TPCANTPConfirmation.PCANTP_N_WRONG_SN
                strTmp = "Wrong Sequence Number"
                Exit Select
            Case Else
                strTmp = "Unknown"
                Exit Select
        End Select
        Return String.Format("{0} ({1})", CByte(result), strTmp)
    End Function
End Module



